<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\FeesType;
use App\Models\FeesCollection;
use App\Models\FeesAssignment;
use App\Models\Student;
use App\Services\PushNotificationService;
use Illuminate\Support\Facades\Log;

class GenerateRecurringFees extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'fees:generate-recurring {--force : Force generation even if not the scheduled day}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate recurring fees for assigned students';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting recurring fee generation...');
        
        $today = now();
        $force = $this->option('force');
        
        // Get all recurring fee types with auto_generate enabled
        $feeTypes = FeesType::where('is_recurring', true)
            ->where('auto_generate', true)
            ->where('is_active', true)
            ->get();
        
        if ($feeTypes->isEmpty()) {
            $this->info('No recurring fee types found.');
            return 0;
        }
        
        $totalGenerated = 0;
        
        foreach ($feeTypes as $feeType) {
            // Check if we should generate today
            $shouldGenerate = $force || $feeType->shouldGenerateNow();
            
            if (!$shouldGenerate) {
                $this->line("Skipping {$feeType->name} - not scheduled for today (Day {$feeType->recurring_day})");
                continue;
            }
            
            $this->info("Processing: {$feeType->name}");
            
            // Get students assigned to this fee type
            $assignments = FeesAssignment::where('fees_type_id', $feeType->id)->get();
            
            if ($assignments->isEmpty()) {
                // If no specific assignments, check if we should apply to all students
                $this->line("  No assignments found for this fee type.");
                continue;
            }
            
            foreach ($assignments as $assignment) {
                // Get students based on assignment (class, section, or specific student)
                $students = $this->getStudentsForAssignment($assignment);
                
                foreach ($students as $student) {
                    // Check if fee already exists for this month/period
                    $existingFee = $this->checkExistingFee($feeType, $student, $today);
                    
                    if ($existingFee) {
                        $this->line("  Fee already exists for {$student->name} this period");
                        continue;
                    }
                    
                    // Create fee collection
                    $fee = FeesCollection::create([
                        'student_id' => $student->id,
                        'class_id' => $student->class_id,
                        'section_id' => $student->section_id,
                        'fees_type_id' => $feeType->id,
                        'amount' => $feeType->amount,
                        'paid_amount' => 0,
                        'payment_date' => $today,
                    ]);
                    
                    $totalGenerated++;
                    $this->line("  Created fee for {$student->name}: ₹{$feeType->amount}");
                    
                    // Send push notification
                    try {
                        $fee->load('student');
                        $pushService = new PushNotificationService();
                        $pushService->notifyFeeGenerated($fee);
                    } catch (\Exception $e) {
                        Log::error("Failed to send notification for student {$student->id}: " . $e->getMessage());
                    }
                }
            }
            
            // Update last generated date
            $feeType->last_generated_date = $today;
            $feeType->save();
        }
        
        $this->info("Completed! Generated {$totalGenerated} fee records.");
        
        return 0;
    }
    
    /**
     * Get students for a fee assignment
     */
    protected function getStudentsForAssignment(FeesAssignment $assignment)
    {
        $query = Student::where('is_active', true);
        
        if ($assignment->student_id) {
            return Student::where('id', $assignment->student_id)->get();
        }
        
        if ($assignment->class_id) {
            $query->where('class_id', $assignment->class_id);
        }
        
        if ($assignment->section_id) {
            $query->where('section_id', $assignment->section_id);
        }
        
        return $query->get();
    }
    
    /**
     * Check if fee already exists for this period
     */
    protected function checkExistingFee(FeesType $feeType, Student $student, $date)
    {
        $query = FeesCollection::where('student_id', $student->id)
            ->where('fees_type_id', $feeType->id);
        
        switch ($feeType->recurring_frequency) {
            case 'monthly':
                $query->whereMonth('created_at', $date->month)
                      ->whereYear('created_at', $date->year);
                break;
            case 'quarterly':
                $quarter = ceil($date->month / 3);
                $startMonth = ($quarter - 1) * 3 + 1;
                $query->whereBetween('created_at', [
                    $date->copy()->month($startMonth)->startOfMonth(),
                    $date->copy()->month($startMonth + 2)->endOfMonth(),
                ]);
                break;
            case 'yearly':
                $query->whereYear('created_at', $date->year);
                break;
        }
        
        return $query->exists();
    }
}
