<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\ExamSchedule;
use App\Models\ExamType;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Subject;
use App\Models\Room;

class ExamScheduleController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examTypeId = $request->input('exam_type_id');
        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');

        $query = ExamSchedule::with(['examType','schoolClass','section','subject','room'])
            ->orderBy('exam_date')
            ->orderBy('start_time');

        if ($examTypeId) {
            $query->where('exam_type_id', $examTypeId);
        }
        if ($classId) {
            $query->where('class_id', $classId);
        }
        if ($sectionId) {
            $query->where('section_id', $sectionId);
        }

        $schedules = $query->withCount('results')->paginate(20);

        $examTypes = ExamType::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();

        return view('admin.exams.schedules.index', [
            'user' => $user,
            'schedules' => $schedules,
            'examTypes' => $examTypes,
            'classes' => $classes,
            'sections' => $sections,
            'examTypeId' => $examTypeId,
            'classId' => $classId,
            'sectionId' => $sectionId,
            'rooms' => $rooms,
        ]);
    }

    public function show(ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examSchedule->load(['examType','schoolClass','section','subject','room','results.student']);

        return view('admin.exams.schedules.show', [
            'user' => $user,
            'examSchedule' => $examSchedule,
        ]);
    }

    public function create(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examTypes = ExamType::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();

        $examTypeId = $request->input('exam_type_id');
        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');

        return view('admin.exams.schedules.create', [
            'user' => $user,
            'examTypes' => $examTypes,
            'classes' => $classes,
            'sections' => $sections,
            'subjects' => $subjects,
            'rooms' => $rooms,
            'examTypeId' => $examTypeId,
            'classId' => $classId,
            'sectionId' => $sectionId,
        ]);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'exam_type_id' => 'required|exists:exam_types,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'exam_date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room_id' => 'nullable|exists:rooms,id',
            'room' => 'nullable|string|max:255',
            'max_marks' => 'nullable|integer|min:0',
            'pass_mark' => 'nullable|integer|min:0',
            'is_published' => 'sometimes|boolean',
            'note' => 'nullable|string',
        ]);

        $roomName = $validated['room'] ?? null;
        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $roomName = $room->name;
            }
        }

        ExamSchedule::create([
            'exam_type_id' => $validated['exam_type_id'],
            'class_id' => $validated['class_id'],
            'section_id' => $validated['section_id'],
            'subject_id' => $validated['subject_id'],
            'exam_date' => $validated['exam_date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'room' => $roomName,
            'room_id' => $validated['room_id'] ?? null,
            'max_marks' => $validated['max_marks'] ?? null,
            'pass_mark' => $validated['pass_mark'] ?? null,
            'is_published' => $request->boolean('is_published'),
            'note' => $validated['note'] ?? null,
        ]);

        return redirect()->route('admin.exam-schedules.index')->with('success', 'Exam schedule created successfully.');
    }

    public function edit(ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examTypes = ExamType::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();

        return view('admin.exams.schedules.edit', [
            'user' => $user,
            'examSchedule' => $examSchedule,
            'examTypes' => $examTypes,
            'classes' => $classes,
            'sections' => $sections,
            'subjects' => $subjects,
            'rooms' => $rooms,
        ]);
    }

    public function update(Request $request, ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'exam_type_id' => 'required|exists:exam_types,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'exam_date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room_id' => 'nullable|exists:rooms,id',
            'room' => 'nullable|string|max:255',
            'max_marks' => 'nullable|integer|min:0',
            'pass_mark' => 'nullable|integer|min:0',
            'is_published' => 'sometimes|boolean',
            'note' => 'nullable|string',
        ]);

        $roomName = $validated['room'] ?? null;
        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $roomName = $room->name;
            }
        }

        $examSchedule->update([
            'exam_type_id' => $validated['exam_type_id'],
            'class_id' => $validated['class_id'],
            'section_id' => $validated['section_id'],
            'subject_id' => $validated['subject_id'],
            'exam_date' => $validated['exam_date'],
            'start_time' => $validated['start_time'],
            'end_time' => $validated['end_time'],
            'room' => $roomName,
            'room_id' => $validated['room_id'] ?? $examSchedule->room_id,
            'max_marks' => $validated['max_marks'] ?? null,
            'pass_mark' => $validated['pass_mark'] ?? $examSchedule->pass_mark,
            'is_published' => $request->boolean('is_published'),
            'note' => $validated['note'] ?? null,
        ]);

        return redirect()->route('admin.exam-schedules.index')->with('success', 'Exam schedule updated successfully.');
    }

    public function destroy(ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examSchedule->delete();

        return redirect()->route('admin.exam-schedules.index')->with('success', 'Exam schedule deleted.');
    }

    /**
     * Show exam results page with class/section selection
     */
    public function results(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();

        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');

        $schedules = collect();
        if ($classId && $sectionId) {
            $schedules = ExamSchedule::with(['examType', 'schoolClass', 'section', 'subject', 'results.student'])
                ->where('class_id', $classId)
                ->where('section_id', $sectionId)
                ->orderBy('exam_date', 'desc')
                ->orderBy('start_time')
                ->get();
        }

        return view('admin.exams.results.index', [
            'user' => $user,
            'classes' => $classes,
            'sections' => $sections,
            'classId' => $classId,
            'sectionId' => $sectionId,
            'schedules' => $schedules,
        ]);
    }
}


