<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\ExamType;

class ExamTypeController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examTypes = ExamType::orderBy('name')->paginate(15);

        return view('admin.exams.types.index', compact('examTypes', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        return view('admin.exams.types.create', compact('user'));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_code' => 'nullable|string|max:50',
            'max_marks' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'is_active' => 'sometimes|boolean',
        ]);

        ExamType::create([
            'name' => $validated['name'],
            'short_code' => $validated['short_code'] ?? null,
            'max_marks' => $validated['max_marks'] ?? null,
            'description' => $validated['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()->route('admin.exam-types.index')->with('success', 'Exam type created successfully.');
    }

    public function edit(ExamType $examType)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        return view('admin.exams.types.edit', compact('examType', 'user'));
    }

    public function update(Request $request, ExamType $examType)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_code' => 'nullable|string|max:50',
            'max_marks' => 'nullable|integer|min:0',
            'description' => 'nullable|string',
            'is_active' => 'sometimes|boolean',
        ]);

        $examType->update([
            'name' => $validated['name'],
            'short_code' => $validated['short_code'] ?? null,
            'max_marks' => $validated['max_marks'] ?? null,
            'description' => $validated['description'] ?? null,
            'is_active' => $request->boolean('is_active'),
        ]);

        return redirect()->route('admin.exam-types.index')->with('success', 'Exam type updated successfully.');
    }

    public function destroy(ExamType $examType)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $examType->delete();

        return redirect()->route('admin.exam-types.index')->with('success', 'Exam type deleted successfully.');
    }
}


