<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\FeesAssignment;
use App\Models\FeesType;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Student;

class FeesAssignmentController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();

        $classId = request('class_id');
        $sectionId = request('section_id');

        $query = FeesAssignment::with(['student', 'feesType'])->orderBy('student_id');

        if ($classId) {
            $query->whereHas('student', function ($q) use ($classId) {
                $q->where('class_id', $classId);
            });
        }
        if ($sectionId) {
            $query->whereHas('student', function ($q) use ($sectionId) {
                $q->where('section_id', $sectionId);
            });
        }

        $assignments = $query->get();

        return view('admin.fees.assignment.index', compact('assignments', 'classes', 'sections', 'classId', 'sectionId', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $feesTypes = FeesType::where('is_active', true)->orderBy('name')->get();

        $classId = request('class_id');
        $sectionId = request('section_id');

        $studentsQuery = Student::orderBy('name');
        if ($classId) {
            $studentsQuery->where('class_id', $classId);
        }
        if ($sectionId) {
            $studentsQuery->where('section_id', $sectionId);
        }
        $students = $studentsQuery->get();

        return view('admin.fees.assignment.create', compact(
            'classes', 'sections', 'feesTypes', 'students', 'classId', 'sectionId', 'user'
        ));
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'fees_type_id' => 'required|exists:fees_types,id',
            'student_ids' => 'required|array|min:1',
            'student_ids.*' => 'exists:students,id',
            'amount' => 'nullable|numeric|min:0',
            'start_date' => 'nullable|date',
        ]);

        $feesType = FeesType::findOrFail($validated['fees_type_id']);
        $amount = $validated['amount'] ?? $feesType->amount;

        foreach ($validated['student_ids'] as $studentId) {
            FeesAssignment::updateOrCreate(
                [
                    'student_id' => $studentId,
                    'fees_type_id' => $feesType->id,
                ],
                [
                    'amount' => $amount,
                    'frequency' => 'monthly',
                    'start_date' => $validated['start_date'] ?? now()->toDateString(),
                    'is_active' => true,
                ]
            );
        }

        return redirect()->route('admin.fees-assignment.index')
            ->with('success', 'Monthly fees assigned to selected students.');
    }

    public function destroy(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $assignment = FeesAssignment::findOrFail($id);
        $assignment->delete();

        return redirect()->route('admin.fees-assignment.index')
            ->with('success', 'Assigned fee removed.');
    }
}



