<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\FeesCollection;
use App\Models\FeePayment;
use App\Models\FeesType;
use App\Models\FeesGroup;
use App\Models\FeesDiscount;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Student;
use App\Services\PushNotificationService;

class FeesCollectionController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collections = FeesCollection::with(['student', 'feesType', 'feesGroup', 'feesDiscount'])
            ->orderByDesc('payment_date')
            ->get();

        return view('admin.fees.collection.index', compact('collections', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $feesTypes = FeesType::where('is_active', true)->orderBy('name')->get();
        $feesDiscounts = FeesDiscount::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        
        // Get students filtered by class/section if provided
        $classId = request()->input('class_id');
        $sectionId = request()->input('section_id');
        $studentsQuery = Student::with(['schoolClass', 'section'])->where('is_active', true);
        
        if ($classId) {
            $studentsQuery->where('class_id', $classId);
        }
        if ($sectionId) {
            $studentsQuery->where('section_id', $sectionId);
        }
        
        $students = $studentsQuery->orderBy('name')->get();

        return view('admin.fees.collection.create', compact(
            'feesTypes', 'feesDiscounts', 'classes', 'sections', 'students', 'user'
        ));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'class_id' => 'nullable|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'fees_type_id' => 'required|exists:fees_types,id',
            'fees_discount_id' => 'nullable|exists:fees_discounts,id',
            'amount' => 'required|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'nullable|string|max:100',
            'reference' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $feeCollection = FeesCollection::create($validated);

        // Send push notification to parent
        try {
            $feeCollection->load('student');
            $pushService = new PushNotificationService();
            
            // If there's a pending amount, notify about fee generated
            if ($feeCollection->amount > $feeCollection->paid_amount) {
                $pushService->notifyFeeGenerated($feeCollection);
            }
            
            // If there's a payment, notify about payment received
            if ($feeCollection->paid_amount > 0) {
                $pushService->notifyPaymentReceived($feeCollection, $feeCollection->paid_amount);
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send fee notification: ' . $e->getMessage());
        }

        return redirect()->route('admin.fees-collection.index')
            ->with('success', 'Fees collected successfully.');
    }

    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with(['student.schoolClass', 'feesType', 'feesDiscount', 'payments.receivedBy'])->findOrFail($id);

        $feesTypes = FeesType::where('is_active', true)->orderBy('name')->get();
        $feesDiscounts = FeesDiscount::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $students = Student::orderBy('name')->get();

        return view('admin.fees.collection.edit', compact(
            'collection', 'feesTypes', 'feesDiscounts', 'classes', 'sections', 'students', 'user'
        ));
    }

    /**
     * Add a new payment to a fee collection
     */
    public function addPayment(Request $request, string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with('student')->findOrFail($id);

        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'payment_method' => 'nullable|string|max:100',
            'reference' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        // Create the payment record
        $payment = FeePayment::create([
            'fees_collection_id' => $collection->id,
            'amount' => $validated['amount'],
            'payment_date' => $validated['payment_date'],
            'payment_method' => $validated['payment_method'] ?? null,
            'reference' => $validated['reference'] ?? null,
            'notes' => $validated['notes'] ?? null,
            'received_by' => $user->id,
        ]);

        // Update the total paid amount on the collection
        $totalPaid = $collection->payments()->sum('amount');
        $collection->paid_amount = $totalPaid;
        $collection->payment_date = $validated['payment_date'];
        $collection->payment_method = $validated['payment_method'] ?? $collection->payment_method;
        $collection->reference = $validated['reference'] ?? $collection->reference;
        $collection->save();

        // Send push notification
        try {
            $pushService = new PushNotificationService();
            $pushService->notifyPaymentReceived($collection, $validated['amount']);
        } catch (\Exception $e) {
            \Log::error('Failed to send payment notification: ' . $e->getMessage());
        }

        return redirect()->route('admin.fees-collection.edit', $collection->id)
            ->with('success', 'Payment of ₹' . number_format($validated['amount'], 2) . ' added successfully.');
    }

    /**
     * Delete a payment from a fee collection
     */
    public function deletePayment(Request $request, string $collectionId, string $paymentId)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::findOrFail($collectionId);
        $payment = FeePayment::where('fees_collection_id', $collectionId)->findOrFail($paymentId);
        
        $payment->delete();

        // Update the total paid amount on the collection
        $totalPaid = $collection->payments()->sum('amount');
        $collection->paid_amount = $totalPaid;
        $collection->save();

        return redirect()->route('admin.fees-collection.edit', $collection->id)
            ->with('success', 'Payment deleted successfully.');
    }

    public function update(Request $request, string $id)
    {
        $collection = FeesCollection::findOrFail($id);
        $oldPaidAmount = $collection->paid_amount;

        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'class_id' => 'nullable|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'fees_type_id' => 'required|exists:fees_types,id',
            'fees_discount_id' => 'nullable|exists:fees_discounts,id',
            'amount' => 'required|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'paid_amount' => 'required|numeric|min:0',
            'payment_date' => 'required|date',
            'payment_method' => 'nullable|string|max:100',
            'reference' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $collection->update($validated);

        // Send notification if payment amount increased (new payment added)
        $newPaidAmount = $validated['paid_amount'];
        if ($newPaidAmount > $oldPaidAmount) {
            try {
                $collection->load('student');
                $pushService = new PushNotificationService();
                $paymentAdded = $newPaidAmount - $oldPaidAmount;
                $pushService->notifyPaymentReceived($collection, $paymentAdded);
            } catch (\Exception $e) {
                \Log::error('Failed to send payment notification: ' . $e->getMessage());
            }
        }

        return redirect()->route('admin.fees-collection.index')
            ->with('success', 'Fees collection updated successfully.');
    }

    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with(['student.schoolClass', 'student.section', 'feesType', 'payments'])->findOrFail($id);

        return view('admin.fees.collection.receipt', compact('collection', 'user'));
    }

    public function destroy(string $id)
    {
        $collection = FeesCollection::findOrFail($id);
        $collection->delete();

        return redirect()->route('admin.fees-collection.index')
            ->with('success', 'Fees collection deleted.');
    }

    /**
     * Search students for autocomplete
     */
    public function searchStudents(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $query = $request->input('q', '');
        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');
        
        $studentsQuery = Student::with(['schoolClass', 'section'])
            ->where('is_active', true);
        
        if ($classId) {
            $studentsQuery->where('class_id', $classId);
        }
        if ($sectionId) {
            $studentsQuery->where('section_id', $sectionId);
        }
        
        if ($query && strlen($query) > 0) {
            $searchTerm = '%' . $query . '%';
            $studentsQuery->where(function($q) use ($searchTerm) {
                $q->whereRaw('LOWER(name) LIKE ?', [strtolower($searchTerm)])
                  ->orWhereRaw('LOWER(admission_no) LIKE ?', [strtolower($searchTerm)])
                  ->orWhereRaw('LOWER(email) LIKE ?', [strtolower($searchTerm)]);
            });
        }
        
        $students = $studentsQuery
            ->orderBy('name')
            ->limit(50)
            ->get()
            ->map(function($student) {
                return [
                    'id' => $student->id,
                    'name' => $student->name,
                    'admission_no' => $student->admission_no,
                    'email' => $student->email,
                    'class' => $student->schoolClass->name ?? '-',
                    'section' => $student->section->name ?? '-',
                ];
            });
        
        return response()->json($students);
    }

    /**
     * Print a single payment receipt
     */
    public function printPayment(string $collectionId, string $paymentId)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with(['student.schoolClass', 'feesType'])->findOrFail($collectionId);
        $payment = FeePayment::with('receivedBy')->where('fees_collection_id', $collectionId)->findOrFail($paymentId);

        return view('admin.fees.collection.print-payment', compact('collection', 'payment', 'user'));
    }

    /**
     * Print all payments for a fee collection
     */
    public function printAllPayments(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with(['student.schoolClass', 'feesType', 'payments.receivedBy'])->findOrFail($id);

        return view('admin.fees.collection.print-all-payments', compact('collection', 'user'));
    }

    /**
     * Print complete receipt (when fully paid)
     */
    public function printComplete(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $collection = FeesCollection::with(['student.schoolClass', 'feesType', 'feesDiscount', 'payments.receivedBy'])->findOrFail($id);

        return view('admin.fees.collection.print-complete', compact('collection', 'user'));
    }
}

