<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\FeesType;

class FeesTypeController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $types = FeesType::orderBy('name')->get();
        return view('admin.fees.type.index', compact('types', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        return view('admin.fees.type.create', compact('user'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_recurring' => 'nullable|boolean',
            'recurring_frequency' => 'nullable|string|in:monthly,quarterly,yearly,one_time',
            'recurring_day' => 'nullable|integer|min:1|max:28',
            'recurring_start_date' => 'nullable|date',
            'recurring_end_date' => 'nullable|date|after_or_equal:recurring_start_date',
            'auto_generate' => 'nullable|boolean',
        ]);
        
        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['is_recurring'] = $request->boolean('is_recurring', false);
        $validated['auto_generate'] = $request->boolean('auto_generate', false);

        FeesType::create($validated);

        return redirect()->route('admin.fees-type.index')->with('success', 'Fees Type created successfully.');
    }

    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        $type = FeesType::findOrFail($id);
        return view('admin.fees.type.edit', compact('type', 'user'));
    }

    public function update(Request $request, string $id)
    {
        $type = FeesType::findOrFail($id);
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'is_active' => 'nullable|boolean',
            'is_recurring' => 'nullable|boolean',
            'recurring_frequency' => 'nullable|string|in:monthly,quarterly,yearly,one_time',
            'recurring_day' => 'nullable|integer|min:1|max:28',
            'recurring_start_date' => 'nullable|date',
            'recurring_end_date' => 'nullable|date|after_or_equal:recurring_start_date',
            'auto_generate' => 'nullable|boolean',
        ]);
        
        $validated['is_active'] = $request->boolean('is_active', true);
        $validated['is_recurring'] = $request->boolean('is_recurring', false);
        $validated['auto_generate'] = $request->boolean('auto_generate', false);

        $type->update($validated);

        return redirect()->route('admin.fees-type.index')->with('success', 'Fees Type updated successfully.');
    }

    public function destroy(string $id)
    {
        $type = FeesType::findOrFail($id);
        $type->delete();

        return redirect()->route('admin.fees-type.index')->with('success', 'Fees Type deleted successfully.');
    }
}


