<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Homework;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Subject;
use App\Models\Teacher;
use App\Models\Student;
use App\Models\HomeworkSubmission;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;

class HomeworkController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::with(['teacher', 'schoolClass', 'section', 'subject', 'submissions'])
            ->orderByDesc('assigned_date')
            ->orderByDesc('created_at')
            ->get();

        return view('admin.homework.index', compact('homework', 'user'));
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $teachers = Teacher::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();

        return view('admin.homework.create', compact('teachers', 'classes', 'sections', 'subjects', 'user'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'teacher_id' => 'required|exists:teachers,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'subject_id' => 'nullable|exists:subjects,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assigned_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:assigned_date',
            'status' => 'nullable|in:active,completed,cancelled',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240', // 10MB max per file
            'notes' => 'nullable|string',
            'is_published' => 'nullable|boolean',
        ]);

        // Handle file uploads
        $attachmentPaths = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('homework/attachments', 'public');
                $attachmentPaths[] = $path;
            }
        }
        $validated['attachments'] = !empty($attachmentPaths) ? $attachmentPaths : null;
        $validated['status'] = $validated['status'] ?? 'active';
        $validated['is_published'] = $validated['is_published'] ?? true;

        Homework::create($validated);

        return redirect()->route('admin.homework.index')
            ->with('success', 'Homework created successfully.');
    }

    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::with(['teacher', 'schoolClass', 'section', 'subject', 'submissions.student'])
            ->findOrFail($id);

        return view('admin.homework.show', compact('homework', 'user'));
    }

    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::findOrFail($id);
        $teachers = Teacher::where('is_active', true)->orderBy('name')->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();

        return view('admin.homework.edit', compact('homework', 'teachers', 'classes', 'sections', 'subjects', 'user'));
    }

    public function update(Request $request, string $id)
    {
        $homework = Homework::findOrFail($id);

        $validated = $request->validate([
            'teacher_id' => 'required|exists:teachers,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'subject_id' => 'nullable|exists:subjects,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'assigned_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:assigned_date',
            'status' => 'nullable|in:active,completed,cancelled',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240',
            'notes' => 'nullable|string',
            'is_published' => 'nullable|boolean',
        ]);

        // Handle new file uploads
        $existingAttachments = $homework->attachments ?? [];
        $newAttachmentPaths = [];
        
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('homework/attachments', 'public');
                $newAttachmentPaths[] = $path;
            }
        }

        // Merge existing and new attachments
        $validated['attachments'] = array_merge($existingAttachments, $newAttachmentPaths);
        if (empty($validated['attachments'])) {
            $validated['attachments'] = null;
        }

        $homework->update($validated);

        return redirect()->route('admin.homework.index')
            ->with('success', 'Homework updated successfully.');
    }

    public function destroy(string $id)
    {
        $homework = Homework::findOrFail($id);
        
        // Delete attached files
        if ($homework->attachments) {
            foreach ($homework->attachments as $attachment) {
                Storage::disk('public')->delete($attachment);
            }
        }
        
        $homework->delete();

        return redirect()->route('admin.homework.index')
            ->with('success', 'Homework deleted successfully.');
    }

    public function getStudents($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::findOrFail($id);

        // Get students for this homework's class and section
        $query = Student::where('class_id', $homework->class_id)
            ->where('is_active', true)
            ->with(['schoolClass', 'section']);

        if ($homework->section_id) {
            $query->where('section_id', $homework->section_id);
        }

        $students = $query->orderBy('name')->get();

        // Get existing submissions with dates
        $existingSubmissions = HomeworkSubmission::where('homework_id', $homework->id)
            ->get()
            ->keyBy('student_id');

        $dueDate = Carbon::parse($homework->due_date);

        return response()->json([
            'students' => $students->map(function($student) use ($existingSubmissions, $dueDate) {
                $submission = $existingSubmissions[$student->id] ?? null;
                $isLate = false;
                if ($submission) {
                    $submittedDate = Carbon::parse($submission->submitted_date);
                    $isLate = $submittedDate->gt($dueDate);
                }
                return [
                    'id' => $student->id,
                    'name' => $student->name,
                    'admission_no' => $student->admission_no,
                    'class' => $student->schoolClass->name ?? '-',
                    'section' => $student->section->name ?? '-',
                    'submitted' => $submission !== null,
                    'submitted_date' => $submission ? $submission->submitted_date->format('d M Y') : null,
                    'is_late' => $isLate,
                    'grade' => $submission ? $submission->grade : null,
                    'teacher_notes' => $submission ? $submission->teacher_notes : null,
                ];
            }),
            'homework' => [
                'id' => $homework->id,
                'title' => $homework->title,
                'due_date' => $homework->due_date->format('Y-m-d'),
            ],
        ]);
    }

    public function markSubmissions(Request $request, $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $homework = Homework::findOrFail($id);

        $validated = $request->validate([
            'student_ids' => 'required|array',
            'student_ids.*' => 'exists:students,id',
            'submitted_date' => 'required|date',
            'grades' => 'nullable|array',
            'grades.*' => 'nullable|string|max:10',
            'teacher_notes' => 'nullable|array',
            'teacher_notes.*' => 'nullable|string|max:1000',
        ]);

        $submittedDate = Carbon::parse($validated['submitted_date']);
        $dueDate = Carbon::parse($homework->due_date);
        $status = $submittedDate->isAfter($dueDate) ? 'late' : 'submitted';

        foreach ($validated['student_ids'] as $studentId) {
            // Verify student belongs to the homework's class/section
            $student = Student::where('id', $studentId)
                ->where('class_id', $homework->class_id)
                ->where(function($query) use ($homework) {
                    if ($homework->section_id) {
                        $query->where('section_id', $homework->section_id);
                    }
                })
                ->first();

            if ($student) {
                // Check if submission already exists
                $existingSubmission = HomeworkSubmission::where('homework_id', $homework->id)
                    ->where('student_id', $studentId)
                    ->first();
                
                if ($existingSubmission) {
                    // Update existing submission - preserve original submission date
                    // Only update grade and notes
                    $updateData = [];
                    
                    // Always allow updating grade
                    if (isset($validated['grades'][$studentId])) {
                        $updateData['grade'] = $validated['grades'][$studentId] ?: null;
                    }
                    
                    // Always allow updating teacher notes
                    if (isset($validated['teacher_notes'][$studentId])) {
                        $updateData['teacher_notes'] = $validated['teacher_notes'][$studentId] ?: null;
                    }
                    
                    if (!empty($updateData)) {
                        $existingSubmission->update($updateData);
                    }
                } else {
                    // Create new submission
                    $updateData = [
                        'submitted_date' => $validated['submitted_date'],
                        'status' => $status,
                    ];
                    
                    // Add grade if provided
                    if (isset($validated['grades'][$studentId]) && !empty($validated['grades'][$studentId])) {
                        $updateData['grade'] = $validated['grades'][$studentId];
                    }
                    
                    // Add teacher notes if provided
                    if (isset($validated['teacher_notes'][$studentId]) && !empty($validated['teacher_notes'][$studentId])) {
                        $updateData['teacher_notes'] = $validated['teacher_notes'][$studentId];
                    }
                    
                    HomeworkSubmission::create(array_merge([
                        'homework_id' => $homework->id,
                        'student_id' => $studentId,
                    ], $updateData));
                }
            }
        }

        return redirect()->route('admin.homework.index')
            ->with('success', 'Submissions marked successfully.');
    }
}
