<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\TeacherNotification;
use App\Services\PushNotificationService;

class LeaveRequestController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $requests = LeaveRequest::with(['user','student','leaveType'])
            ->orderByDesc('created_at')
            ->paginate(15);

        return view('admin.leave-requests.index', compact('requests', 'user'));
    }

    public function edit(LeaveRequest $leaveRequest)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $leaveTypes = LeaveType::orderBy('name')->get();
        return view('admin.leave-requests.edit', compact('leaveRequest', 'leaveTypes', 'user'));
    }

    public function update(Request $request, LeaveRequest $leaveRequest)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'status' => 'required|in:pending,approved,rejected',
            'admin_remarks' => 'nullable|string',
            'leave_type_id' => 'nullable|exists:leave_types,id',
        ]);

        $oldStatus = $leaveRequest->status;
        
        $leaveRequest->update([
            'status' => $validated['status'],
            'admin_remarks' => $validated['admin_remarks'] ?? null,
            'leave_type_id' => $validated['leave_type_id'] ?? $leaveRequest->leave_type_id,
        ]);

        // Send push notification if status changed
        if ($oldStatus !== $validated['status'] && in_array($validated['status'], ['approved', 'rejected'])) {
            try {
                $leaveRequest->load('student');
                
                // If it's a parent's leave request, send push notification
                if ($leaveRequest->role === 'parent' && $leaveRequest->student) {
                    $pushService = new PushNotificationService();
                    $pushService->notifyLeaveStatusUpdate($leaveRequest);
                }
                
                // If it's a teacher's leave request, send teacher notification
                if ($leaveRequest->role === 'teacher') {
                    TeacherNotification::notifyLeaveStatus($leaveRequest);
                }
            } catch (\Exception $e) {
                \Log::error('Failed to send leave notification: ' . $e->getMessage());
            }
        }

        return redirect()->route('admin.leave-requests.index')->with('success', 'Leave request updated.');
    }

    public function destroy(LeaveRequest $leaveRequest)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $leaveRequest->delete();

        return redirect()->route('admin.leave-requests.index')->with('success', 'Leave request deleted.');
    }
}


