<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Section;
use App\Models\SchoolClass;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class SectionController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $sections = Section::with('schoolClass')->withCount('students')->orderBy('name')->get();
        return view('admin.section.index', compact('sections', 'user'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        return view('admin.section.create', compact('classes', 'user'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'class_id' => 'nullable|exists:school_classes,id',
            'seats' => 'required|integer|min:1|max:1000',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        Section::create($validated);

        return redirect()->route('admin.section.index')
            ->with('success', 'Section created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $section = Section::findOrFail($id);
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        return view('admin.section.edit', compact('section', 'classes', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $section = Section::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'class_id' => 'nullable|exists:school_classes,id',
            'seats' => 'required|integer|min:1|max:1000',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $section->update($validated);

        return redirect()->route('admin.section.index')
            ->with('success', 'Section updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $section = Section::findOrFail($id);
        $section->delete();

        return redirect()->route('admin.section.index')
            ->with('success', 'Section deleted successfully.');
    }
}
