<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Student;
use App\Models\AcademicYear;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Batch;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class StudentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $students = Student::with(['academicYear', 'schoolClass', 'section', 'batch', 'parent'])
            ->orderBy('created_at', 'desc')
            ->get();
        
        return view('admin.student.index', compact('students', 'user'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $academicYears = AcademicYear::where('is_active', true)->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        
        // Get sections with available seats (seats not filled)
        $sections = Section::where('is_active', true)
            ->withCount(['students' => function($query) {
                $query->where('is_active', true);
            }])
            ->get()
            ->filter(function($section) {
                $filledSeats = $section->students_count ?? 0;
                $totalSeats = $section->seats ?? 20;
                return $filledSeats < $totalSeats; // Only sections with available seats
            })
            ->sortBy('name')
            ->values();
        
        $batches = Batch::where('is_active', true)->orderBy('name')->get();
        $parents = User::where('role', 'parent')->with('students')->orderBy('name')->get();
        
        return view('admin.student.create', compact('academicYears', 'classes', 'sections', 'batches', 'parents', 'user'));
    }

    /**
     * Search parents for autocomplete
     */
    public function searchParents(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            return response()->json(['error' => 'Unauthorized'], 403);
        }

        $query = $request->input('q', '');
        $parentId = $request->input('id');
        
        $parentsQuery = User::where('role', 'parent');
        
        // If specific ID requested, return that parent
        if ($parentId) {
            $parentsQuery->where('id', $parentId);
        } elseif ($query && strlen($query) > 0) {
            // If query provided, search (case-insensitive)
            $searchTerm = '%' . $query . '%';
            $parentsQuery->where(function($q) use ($searchTerm) {
                $q->whereRaw('LOWER(name) LIKE ?', [strtolower($searchTerm)])
                  ->orWhereRaw('LOWER(email) LIKE ?', [strtolower($searchTerm)])
                  ->orWhere('phone', 'like', $searchTerm);
            });
        } else {
            // If no query, return empty (don't show all parents)
            return response()->json([]);
        }
        
        $parents = $parentsQuery
            ->withCount('students')
            ->orderBy('name')
            ->limit(20)
            ->get()
            ->map(function($parent) {
                return [
                    'id' => $parent->id,
                    'name' => $parent->name,
                    'email' => $parent->email,
                    'phone' => $parent->phone,
                    'occupation' => $parent->occupation,
                    'address' => $parent->address,
                    'students_count' => $parent->students_count,
                ];
            });
        
        return response()->json($parents);
    }

    /**
     * Generate unique admission number
     */
    private function generateAdmissionNo()
    {
        $year = date('Y');
        $prefix = 'ADM-' . $year . '-';
        
        // Get the last admission number for this year
        $lastStudent = Student::where('admission_no', 'like', $prefix . '%')
            ->orderBy('admission_no', 'desc')
            ->first();
        
        if ($lastStudent) {
            // Extract the number part and increment
            $lastNumber = (int) substr($lastStudent->admission_no, strlen($prefix));
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }
        
        // Format: ADM-2026-0001
        return $prefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Check if existing parent is selected
        $existingParentId = $request->input('existing_parent_id');
        
        // Basic validation for admission form
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'date_of_birth' => 'required|date',
            'gender' => 'required|in:Male,Female,Other',
            'photo' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'academic_year_id' => 'required|exists:academic_years,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => [
                'required',
                'exists:sections,id',
                function ($attribute, $value, $fail) {
                    $section = Section::find($value);
                    if ($section) {
                        $filledSeats = $section->students()->where('is_active', true)->count();
                        $totalSeats = $section->seats ?? 20;
                        if ($filledSeats >= $totalSeats) {
                            $fail('The selected section is full. No available seats.');
                        }
                    }
                },
            ],
            'batch_id' => 'nullable|exists:batches,id',
            'previous_school_name' => 'nullable|string|max:255',
            'previous_school_address' => 'nullable|string',
            'blood_group' => 'nullable|string|max:255',
            'height' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'medical_care_needed' => 'nullable|in:yes,no',
            'medical_care_details' => 'nullable|string',
            'existing_parent_id' => 'nullable|exists:users,id',
        ];

        // Only require new parent fields if no existing parent is selected
        if (!$existingParentId) {
            $rules['parent_name'] = 'required|string|max:255';
            $rules['parent_email'] = 'required|email|max:255|unique:users,email';
            $rules['parent_password'] = 'required|string|min:6';
        }
        
        $rules['parent_phone'] = 'nullable|string|max:20';
        $rules['parent_occupation'] = 'nullable|string|max:255';
        $rules['parent_address'] = 'nullable|string';
        $rules['parent_photo'] = 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048';

        $validated = $request->validate($rules);

        // Auto-generate admission number
        $validated['admission_no'] = $this->generateAdmissionNo();

        // Handle student photo upload
        if ($request->hasFile('photo')) {
            $validated['photo'] = $request->file('photo')->store('students/photos', 'public');
        }

        // Handle parent - either use existing or create new
        if ($existingParentId) {
            // Use existing parent
            $parentUser = User::findOrFail($existingParentId);
            $validated['parent_id'] = $parentUser->id;
            
            // Store parent details from existing parent
            $validated['guardian_name'] = $parentUser->name;
            $validated['guardian_phone'] = $parentUser->phone ?? null;
            $validated['guardian_occupation'] = $parentUser->occupation ?? null;
            $validated['guardian_address'] = $parentUser->address ?? null;
            
            // Update parent details if provided
            if ($request->filled('parent_phone')) {
                $parentUser->phone = $request->input('parent_phone');
            }
            if ($request->filled('parent_occupation')) {
                $parentUser->occupation = $request->input('parent_occupation');
            }
            if ($request->filled('parent_address')) {
                $parentUser->address = $request->input('parent_address');
            }
            if ($request->hasFile('parent_photo')) {
                $parentPhotoPath = $request->file('parent_photo')->store('users/photos', 'public');
                $parentUser->photo = $parentPhotoPath;
            }
            $parentUser->save();
        } else {
            // Create new parent user account
            $parentUser = User::create([
                'name' => $validated['parent_name'],
                'email' => $validated['parent_email'],
                'password' => Hash::make($validated['parent_password']),
                'role' => 'parent',
                'phone' => $validated['parent_phone'] ?? null,
                'occupation' => $validated['parent_occupation'] ?? null,
                'address' => $validated['parent_address'] ?? null,
            ]);

            // Link student to parent
            $validated['parent_id'] = $parentUser->id;

            // Handle parent photo upload
            if ($request->hasFile('parent_photo')) {
                $parentPhotoPath = $request->file('parent_photo')->store('users/photos', 'public');
                $parentUser->photo = $parentPhotoPath;
                $parentUser->save();
            }

            // Store parent details in student record
            $validated['guardian_name'] = $validated['parent_name'];
            $validated['guardian_phone'] = $validated['parent_phone'] ?? null;
            $validated['guardian_occupation'] = $validated['parent_occupation'] ?? null;
            $validated['guardian_address'] = $validated['parent_address'] ?? null;
        }

        // Handle medical care details - store in student_details if medical care is needed
        if ($request->input('medical_care_needed') === 'yes' && $request->filled('medical_care_details')) {
            $validated['student_details'] = 'Medical Care Required: ' . $request->input('medical_care_details');
        }

        // Remove parent fields that are not in students table
        if (isset($validated['parent_email'])) unset($validated['parent_email']);
        if (isset($validated['parent_password'])) unset($validated['parent_password']);
        if (isset($validated['parent_name'])) unset($validated['parent_name']);
        if (isset($validated['parent_phone'])) unset($validated['parent_phone']);
        if (isset($validated['parent_occupation'])) unset($validated['parent_occupation']);
        if (isset($validated['parent_address'])) unset($validated['parent_address']);
        if (isset($validated['parent_photo'])) unset($validated['parent_photo']);
        if (isset($validated['existing_parent_id'])) unset($validated['existing_parent_id']);
        unset($validated['medical_care_needed']);
        unset($validated['medical_care_details']);

        // Set default values
        $validated['is_active'] = true;

        // Create student
        $student = Student::create($validated);

        return redirect()->route('admin.fees-collection.create', ['student_id' => $student->id])
            ->with('success', 'Student created successfully with admission number: ' . $validated['admission_no'] . '. Parent account created with email: ' . $request->input('parent_email') . '. You can now add recurring fees for this student.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $student = Student::with(['academicYear', 'schoolClass', 'section', 'batch', 'parent'])->findOrFail($id);
        return view('admin.student.show', compact('student', 'user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $student = Student::findOrFail($id);
        $academicYears = AcademicYear::where('is_active', true)->get();
        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $batches = Batch::where('is_active', true)->orderBy('name')->get();
        $parents = User::where('role', 'parent')->orderBy('name')->get();
        
        return view('admin.student.edit', compact('student', 'academicYears', 'classes', 'sections', 'batches', 'parents', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $student = Student::findOrFail($id);
        
        $rules = [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'category' => 'nullable|string|max:255',
            'roll_number' => 'nullable|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'academic_year_id' => 'nullable|exists:academic_years,id',
            'class_id' => 'nullable|exists:school_classes,id',
            'section_id' => 'nullable|exists:sections,id',
            'batch_id' => 'nullable|exists:batches,id',
            'parent_id' => 'nullable|exists:users,id',
            'parent_email' => 'nullable|email|max:255|unique:users,email',
            'parent_password' => 'nullable|string|min:6',
            'parent_name' => 'nullable|string|max:255',
            'father_name' => 'nullable|string|max:255',
            'father_phone' => 'nullable|string|max:20',
            'father_occupation' => 'nullable|string|max:255',
            'father_photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'mother_name' => 'nullable|string|max:255',
            'mother_phone' => 'nullable|string|max:20',
            'mother_occupation' => 'nullable|string|max:255',
            'mother_photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'guardian_name' => 'nullable|string|max:255',
            'guardian_phone' => 'nullable|string|max:20',
            'guardian_relation' => 'nullable|string|max:255',
            'guardian_photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'address' => 'nullable|string',
            'city' => 'nullable|string|max:255',
            'state' => 'nullable|string|max:255',
            'zip_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|max:255',
            'is_active' => 'boolean',
        ];

        // If parent email is provided, password is required and vice versa
        if ($request->filled('parent_email') || $request->filled('parent_password')) {
            $rules['parent_email'] = 'required|email|max:255|unique:users,email';
            $rules['parent_password'] = 'required|string|min:6';
            $rules['parent_name'] = 'required|string|max:255';
        }

        $validated = $request->validate($rules);

        // Create parent user account if email and password provided (and not already linked)
        $parentId = $validated['parent_id'] ?? $student->parent_id;
        
        if (!$parentId && $request->filled('parent_email') && $request->filled('parent_password')) {
            // Create new parent user
            $parentName = $request->input('parent_name') ?: $request->input('father_name') ?: $request->input('mother_name') ?: 'Parent';
            
            $parentUser = User::create([
                'name' => $parentName,
                'email' => $request->input('parent_email'),
                'password' => Hash::make($request->input('parent_password')),
                'role' => 'parent',
            ]);
            
            $parentId = $parentUser->id;
            $validated['parent_id'] = $parentId;
        }

        // Handle file uploads
        if ($request->hasFile('photo')) {
            if ($student->photo) {
                Storage::disk('public')->delete($student->photo);
            }
            $validated['photo'] = $request->file('photo')->store('students/photos', 'public');
        }
        if ($request->hasFile('father_photo')) {
            if ($student->father_photo) {
                Storage::disk('public')->delete($student->father_photo);
            }
            $validated['father_photo'] = $request->file('father_photo')->store('students/parents', 'public');
        }
        if ($request->hasFile('mother_photo')) {
            if ($student->mother_photo) {
                Storage::disk('public')->delete($student->mother_photo);
            }
            $validated['mother_photo'] = $request->file('mother_photo')->store('students/parents', 'public');
        }
        if ($request->hasFile('guardian_photo')) {
            if ($student->guardian_photo) {
                Storage::disk('public')->delete($student->guardian_photo);
            }
            $validated['guardian_photo'] = $request->file('guardian_photo')->store('students/parents', 'public');
        }

        // Remove parent_email and parent_password from validated (not in students table)
        unset($validated['parent_email']);
        unset($validated['parent_password']);
        unset($validated['parent_name']);

        $student->update($validated);

        $message = 'Student updated successfully';
        if ($parentId && !$request->filled('parent_id') && $student->parent_id != $parentId) {
            $message .= '. Parent account created with email: ' . $request->input('parent_email');
        }

        return redirect()->route('admin.student.index')
            ->with('success', $message);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $student = Student::findOrFail($id);
        
        // Delete associated photos
        if ($student->photo) {
            Storage::disk('public')->delete($student->photo);
        }
        if ($student->father_photo) {
            Storage::disk('public')->delete($student->father_photo);
        }
        if ($student->mother_photo) {
            Storage::disk('public')->delete($student->mother_photo);
        }
        if ($student->guardian_photo) {
            Storage::disk('public')->delete($student->guardian_photo);
        }
        
        $student->delete();

        return redirect()->route('admin.student.index')
            ->with('success', 'Student deleted successfully.');
    }
}
