<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Teacher;
use App\Models\TeacherDocument;
use App\Models\SchoolClass;
use App\Models\User;
use App\Models\Subject;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class TeacherController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $teachers = Teacher::with(['subjects', 'documents'])->orderBy('created_at', 'desc')->get();
        return view('admin.teacher.index', compact('teachers', 'user'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        return view('admin.teacher.create', compact('user', 'subjects'));
    }

    /**
     * Generate unique teacher ID
     */
    private function generateTeacherId()
    {
        $year = date('Y');
        $prefix = 'TCH-' . $year . '-';
        
        $lastTeacher = Teacher::where('teacher_id', 'like', $prefix . '%')
            ->orderBy('teacher_id', 'desc')
            ->first();
        
        if ($lastTeacher) {
            $lastNumber = (int) substr($lastTeacher->teacher_id, strlen($prefix));
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }
        
        return $prefix . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:teachers,email',
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'subject' => 'nullable|string|max:255',
            'father_name' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'marital_status' => 'nullable|in:Married,Unmarried',
            'contract_type' => 'nullable|in:Contractual,Hourly',
            'shift' => 'nullable|in:Day Shift,Night Shift',
            'work_location' => 'nullable|string|max:255',
            'join_date' => 'nullable|date',
            'experience' => 'nullable|string|max:255',
            'qualification' => 'nullable|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'blood_group' => 'nullable|string|max:255',
            'height' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:255',
            'bank_name' => 'nullable|string|max:255',
            'ifsc_code' => 'nullable|string|max:255',
            'national_identification_number' => 'nullable|string|max:255',
            'previous_school_name' => 'nullable|string|max:255',
            'previous_school_address' => 'nullable|string',
            'password' => 'required|string|min:6',
            'documents' => 'nullable|array',
            'documents.*.name' => 'nullable|string|max:255',
            'documents.*.file' => 'nullable|file|mimes:pdf,doc,docx,jpg,jpeg,png|max:5120',
            'subjects' => 'nullable|array',
            'subjects.*' => 'exists:subjects,id',
        ]);

        // Auto-generate teacher ID
        $validated['teacher_id'] = $this->generateTeacherId();

        // Handle photo upload
        if ($request->hasFile('photo')) {
            $validated['photo'] = $request->file('photo')->store('teachers/photos', 'public');
        }

        // Create user account for teacher
        $user = User::create([
            'name' => $validated['name'],
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'role' => 'teacher',
        ]);

        $validated['user_id'] = $user->id;
        unset($validated['password']);

        // Create teacher
        $teacher = Teacher::create($validated);

        // Attach subjects (many-to-many)
        $teacher->subjects()->sync($request->input('subjects', []));

        // Handle document uploads
        if ($request->has('documents')) {
            foreach ($request->input('documents') as $index => $docData) {
                // Skip if no file uploaded for this row
                if (!$request->hasFile("documents.{$index}.file")) {
                    continue;
                }

                $file = $request->file("documents.{$index}.file");
                $filePath = $file->store('teachers/documents', 'public');

                TeacherDocument::create([
                    'teacher_id' => $teacher->id,
                    'document_name' => $docData['name'] ?? 'Document',
                    'file_path' => $filePath,
                    'file_type' => $file->getClientOriginalExtension(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        return redirect()->route('admin.teacher.index')
            ->with('success', 'Teacher created successfully with ID: ' . $validated['teacher_id']);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $teacher = Teacher::with(['schoolClass', 'documents', 'user'])->findOrFail($id);
        return view('admin.teacher.show', compact('teacher', 'user'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }
        
        $teacher = Teacher::with('documents', 'subjects')->findOrFail($id);
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        return view('admin.teacher.edit', compact('teacher', 'subjects', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $teacher = Teacher::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:teachers,email,' . $id,
            'phone' => 'nullable|string|max:20',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:Male,Female,Other',
            'subject' => 'nullable|string|max:255',
            'father_name' => 'nullable|string|max:255',
            'mother_name' => 'nullable|string|max:255',
            'marital_status' => 'nullable|in:Married,Unmarried',
            'contract_type' => 'nullable|in:Contractual,Hourly',
            'shift' => 'nullable|in:Day Shift,Night Shift',
            'work_location' => 'nullable|string|max:255',
            'join_date' => 'nullable|date',
            'experience' => 'nullable|string|max:255',
            'qualification' => 'nullable|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'blood_group' => 'nullable|string|max:255',
            'height' => 'nullable|string|max:255',
            'weight' => 'nullable|string|max:255',
            'bank_account_number' => 'nullable|string|max:255',
            'bank_name' => 'nullable|string|max:255',
            'ifsc_code' => 'nullable|string|max:255',
            'national_identification_number' => 'nullable|string|max:255',
            'previous_school_name' => 'nullable|string|max:255',
            'previous_school_address' => 'nullable|string',
            'password' => 'nullable|string|min:6',
            'documents' => 'nullable|array',
            'documents.*.name' => 'nullable|string|max:255',
            'documents.*.file' => 'nullable|file|mimes:pdf,doc,docx,jpg,jpeg,png|max:5120',
            'subjects' => 'nullable|array',
            'subjects.*' => 'exists:subjects,id',
        ]);

        // Handle photo upload
        if ($request->hasFile('photo')) {
            // Delete old photo if exists
            if ($teacher->photo) {
                Storage::disk('public')->delete($teacher->photo);
            }
            $validated['photo'] = $request->file('photo')->store('teachers/photos', 'public');
        }

        // Update user account if password is provided
        if ($request->filled('password') && $teacher->user) {
            $teacher->user->update([
                'name' => $validated['name'],
                'email' => $validated['email'],
                'password' => Hash::make($validated['password']),
            ]);
        } elseif ($teacher->user) {
            $teacher->user->update([
                'name' => $validated['name'],
                'email' => $validated['email'],
            ]);
        }

        unset($validated['password']);

        // Update teacher
        $teacher->update($validated);

        // Sync subjects
        $teacher->subjects()->sync($request->input('subjects', []));

        // Handle new document uploads
        if ($request->has('documents')) {
            foreach ($request->input('documents') as $index => $docData) {
                // Skip if no file uploaded for this row
                if (!$request->hasFile("documents.{$index}.file")) {
                    continue;
                }

                $file = $request->file("documents.{$index}.file");
                $filePath = $file->store('teachers/documents', 'public');

                TeacherDocument::create([
                    'teacher_id' => $teacher->id,
                    'document_name' => $docData['name'] ?? 'Document',
                    'file_path' => $filePath,
                    'file_type' => $file->getClientOriginalExtension(),
                    'file_size' => $file->getSize(),
                ]);
            }
        }

        return redirect()->route('admin.teacher.show', $teacher->id)
            ->with('success', 'Teacher updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $teacher = Teacher::findOrFail($id);
        
        // Delete documents
        foreach ($teacher->documents as $document) {
            Storage::disk('public')->delete($document->file_path);
        }
        $teacher->documents()->delete();
        
        // Delete photo
        if ($teacher->photo) {
            Storage::disk('public')->delete($teacher->photo);
        }
        
        // Delete user account if exists
        if ($teacher->user) {
            $teacher->user->delete();
        }
        
        $teacher->delete();

        return redirect()->route('admin.teacher.index')
            ->with('success', 'Teacher deleted successfully.');
    }

    /**
     * Delete a specific document
     */
    public function deleteDocument(string $teacherId, string $documentId)
    {
        $teacher = Teacher::findOrFail($teacherId);
        $document = TeacherDocument::where('teacher_id', $teacherId)->findOrFail($documentId);
        
        Storage::disk('public')->delete($document->file_path);
        $document->delete();
        
        return redirect()->route('admin.teacher.show', $teacher->id)
            ->with('success', 'Document deleted successfully.');
    }
}
