<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\TimetableOverride;
use App\Models\Timetable;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Subject;
use App\Models\Teacher;
use App\Models\Room;
use Carbon\Carbon;

class TimetableOverrideController extends Controller
{
    /**
     * Show weekly timetable viewer by class & section.
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $teachers = Teacher::orderBy('name')->get();
        $rooms = Timetable::select('room')
            ->whereNotNull('room')
            ->distinct()
            ->orderBy('room')
            ->pluck('room');

        $classId = $request->input('class_id');
        $sectionId = $request->input('section_id');
        $dayFilter = strtolower($request->input('day', ''));
        $teacherId = $request->input('teacher_id');
        $roomFilter = $request->input('room');

        $slots = collect();
        if ($classId && $sectionId) {
            $query = Timetable::with(['schoolClass','section','subject','teacher'])
                ->where('class_id', $classId)
                ->where('section_id', $sectionId);

            if ($teacherId) {
                $query->where('teacher_id', $teacherId);
            }

            if ($roomFilter) {
                $query->where('room', 'like', '%'.$roomFilter.'%');
            }

            $slots = $query->orderBy('day_of_week')
                ->orderBy('start_time')
                ->get();
        }

        return view('admin.timetable-override.index', [
            'user' => $user,
            'classes' => $classes,
            'sections' => $sections,
            'teachers' => $teachers,
            'classId' => $classId,
            'sectionId' => $sectionId,
            'slots' => $slots,
            'dayFilter' => $dayFilter,
            'teacherId' => $teacherId,
            'roomFilter' => $roomFilter,
            'rooms' => $rooms,
        ]);
    }

    protected function mapDayOfWeek(string $day): int
    {
        $map = [
            'sunday' => 0,
            'monday' => 1,
            'tuesday' => 2,
            'wednesday' => 3,
            'thursday' => 4,
            'friday' => 5,
            'saturday' => 6,
        ];

        return $map[strtolower($day)] ?? 0;
    }

    /**
     * Show the form for creating a new override.
     */
    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $teachers = Teacher::with('subjects')->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();
        $timetables = Timetable::with(['schoolClass','section','subject','teacher'])
            ->orderBy('day_of_week')
            ->orderBy('start_time')
            ->get();
        $days = ['monday','tuesday','wednesday','thursday','friday','saturday','sunday'];
        $date = request('date');

        return view('admin.timetable-override.create', compact(
            'classes', 'sections', 'subjects', 'teachers', 'rooms', 'timetables', 'days', 'date', 'user'
        ));
    }

    /**
     * Store a newly created override in storage.
     */
    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'date' => 'required|date',
            'base_timetable_id' => 'nullable|exists:timetables,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'teacher_id' => 'required|exists:teachers,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room' => 'nullable|string|max:255',
            'room_id' => 'nullable|exists:rooms,id',
            'notes' => 'nullable|string',
        ]);

        $date = Carbon::parse($validated['date'])->toDateString();
        $validated['date'] = $date;

        // Conflict check against overrides for that date
        $conflictOverride = TimetableOverride::whereDate('date', $date)
            ->where('teacher_id', $validated['teacher_id'])
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflictOverride) {
            return back()->withInput()->withErrors(['teacher_id' => 'Teacher already has an override at this time for the selected date.']);
        }

        // Conflict check against base timetable for that day (unless replacing the same base slot)
        $baseQuery = Timetable::where('teacher_id', $validated['teacher_id'])
            ->where('day_of_week', $validated['day_of_week']);

        if (!empty($validated['base_timetable_id'])) {
            $baseQuery->where('id', '!=', $validated['base_timetable_id']);
        }

        $conflictBase = $baseQuery
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflictBase) {
            return back()->withInput()->withErrors(['teacher_id' => 'Teacher already has a base timetable slot at this time on this weekday.']);
        }

        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $validated['room'] = $room->name;
            }
        }
        unset($validated['room_id']);

        TimetableOverride::create($validated);

        return redirect()->route('admin.timetable-override.index')
            ->with('success', 'Override created successfully for the selected date.');
    }

    /**
     * Show the form for editing the specified override.
     */
    public function edit(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $override = TimetableOverride::findOrFail($id);

        $classes = SchoolClass::where('is_active', true)->orderBy('name')->get();
        $sections = Section::where('is_active', true)->orderBy('name')->get();
        $subjects = Subject::where('is_active', true)->orderBy('name')->get();
        $teachers = Teacher::with('subjects')->orderBy('name')->get();
        $rooms = Room::where('is_active', true)->orderBy('name')->get();
        $timetables = Timetable::with(['schoolClass','section','subject','teacher'])
            ->orderBy('day_of_week')
            ->orderBy('start_time')
            ->get();
        $days = ['monday','tuesday','wednesday','thursday','friday','saturday','sunday'];

        return view('admin.timetable-override.edit', compact(
            'override', 'classes', 'sections', 'subjects', 'teachers', 'rooms', 'timetables', 'days', 'user'
        ));
    }

    /**
     * Update the specified override in storage.
     */
    public function update(Request $request, string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $override = TimetableOverride::findOrFail($id);

        $validated = $request->validate([
            'date' => 'required|date',
            'base_timetable_id' => 'nullable|exists:timetables,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'teacher_id' => 'required|exists:teachers,id',
            'day_of_week' => 'required|in:monday,tuesday,wednesday,thursday,friday,saturday,sunday',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room' => 'nullable|string|max:255',
            'room_id' => 'nullable|exists:rooms,id',
            'notes' => 'nullable|string',
        ]);

        $date = Carbon::parse($validated['date'])->toDateString();
        $validated['date'] = $date;

        // Conflict check against overrides for that date (excluding current)
        $conflictOverride = TimetableOverride::whereDate('date', $date)
            ->where('teacher_id', $validated['teacher_id'])
            ->where('id', '!=', $override->id)
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflictOverride) {
            return back()->withInput()->withErrors(['teacher_id' => 'Teacher already has an override at this time for the selected date.']);
        }

        // Conflict check against base timetable for that day (unless replacing the same base slot)
        $baseQuery = Timetable::where('teacher_id', $validated['teacher_id'])
            ->where('day_of_week', $validated['day_of_week']);

        if (!empty($validated['base_timetable_id'])) {
            $baseQuery->where('id', '!=', $validated['base_timetable_id']);
        }

        $conflictBase = $baseQuery
            ->where(function ($q) use ($validated) {
                $start = $validated['start_time'];
                $end = $validated['end_time'];

                $q->whereBetween('start_time', [$start, $end])
                  ->orWhereBetween('end_time', [$start, $end])
                  ->orWhere(function ($qq) use ($start, $end) {
                      $qq->where('start_time', '<=', $start)
                         ->where('end_time', '>=', $end);
                  });
            })
            ->exists();

        if ($conflictBase) {
            return back()->withInput()->withErrors(['teacher_id' => 'Teacher already has a base timetable slot at this time on this weekday.']);
        }

        if (!empty($validated['room_id'])) {
            $room = Room::find($validated['room_id']);
            if ($room) {
                $validated['room'] = $room->name;
            }
        }
        unset($validated['room_id']);

        $override->update($validated);

        return redirect()->route('admin.timetable-override.index')
            ->with('success', 'Override updated successfully for the selected date.');
    }

    /**
     * Remove the specified override from storage.
     */
    public function destroy(string $id)
    {
        $user = Auth::user();
        if (!$user || !$user->isAdmin()) {
            abort(403, 'Unauthorized access');
        }

        $override = TimetableOverride::findOrFail($id);
        $override->delete();

        return redirect()->route('admin.timetable-override.index')
            ->with('success', 'Override deleted successfully.');
    }
}
