<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Models\User;
use Illuminate\Support\Str;

class AuthController extends Controller
{
    /**
     * Login user and return API token
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $user = Auth::user();
            
            $requestedRole = $request->input('role', 'parent');
            
            // Check based on requested role
            if ($requestedRole === 'teacher') {
                if (!$user->isTeacher()) {
                    Auth::logout();
                    return response()->json([
                        'success' => false,
                        'message' => 'Access denied. This app is for teachers only.'
                    ], 403);
                }
            } else {
                // Default to parent check
                if (!$user->isParent()) {
                    Auth::logout();
                    return response()->json([
                        'success' => false,
                        'message' => 'Access denied. This app is for parents only.'
                    ], 403);
                }
            }

            // Generate API token (simple token for now, can be upgraded to Sanctum later)
            $token = bin2hex(random_bytes(32));
            
            // Store token in user's remember_token for now (temporary solution)
            $user->remember_token = hash('sha256', $token);
            $user->save();

            return response()->json([
                'success' => true,
                'message' => 'Login successful',
                'data' => [
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'phone' => $user->phone,
                        'photo' => $user->photo ? asset('storage/' . $user->photo) : null,
                    ],
                    'token' => $token,
                ]
            ], 200);
        }

        return response()->json([
            'success' => false,
            'message' => 'Invalid email or password'
        ], 401);
    }

    /**
     * Get authenticated user
     */
    public function user(Request $request)
    {
        $user = $request->user();
        
        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'photo' => $user->photo ? asset('storage/' . $user->photo) : null,
            ]
        ], 200);
    }

    /**
     * Logout user
     */
    public function logout(Request $request)
    {
        $user = $request->user();
        $user->remember_token = null;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully'
        ], 200);
    }
}
