<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Teacher;
use App\Models\Student;
use App\Models\Homework;
use App\Models\Attendance;
use Illuminate\Support\Facades\Log;

class TeacherController extends Controller
{
    /**
     * Get teacher dashboard data
     */
    public function dashboard(Request $request)
    {
        $user = $request->user();
        
        // Get teacher profile
        $teacher = Teacher::where('user_id', $user->id)->first();
        
        if (!$teacher) {
            return response()->json([
                'success' => false,
                'message' => 'Teacher profile not found for this user.'
            ], 404);
        }

        // Stats
        // 1. Total Students (in classes taught by this teacher)
        // This is complex if not directly linked. For now, we can count total students or those in teacher's subjects/sections.
        // Simplified: Count students where teacher has assigned homework (as a proxy) or if there's a direct class link.
        // Teacher model has `schoolClass` (belongsTo class_id). If they are a class teacher:
        $classStudentCount = 0;
        if ($teacher->class_id) {
            $classStudentCount = Student::where('class_id', $teacher->class_id)->count();
        }
        
        // 2. Homeworks created
        $homeworkCount = Homework::where('teacher_id', $teacher->id)->count();

        return response()->json([
            'success' => true,
            'data' => [
                'profile' => [
                    'id' => $teacher->id,
                    'name' => $teacher->name,
                    'email' => $teacher->email,
                    'photo' => $teacher->photo ? asset('storage/' . $teacher->photo) : null,
                    'subject' => $teacher->subject,
                ],
                'stats' => [
                    'my_students' => $classStudentCount,
                    'homework_assigned' => $homeworkCount,
                ],
                'recent_homeworks' => Homework::where('teacher_id', $teacher->id)
                    ->orderByDesc('created_at')
                    ->take(5)
                    ->get()
                    ->map(function($hw) {
                        return [
                            'id' => $hw->id,
                            'title' => $hw->title,
                            'class' => $hw->schoolClass->name ?? 'N/A',
                            'due_date' => $hw->due_date ? $hw->due_date->format('d M Y') : 'N/A',
                        ];
                    }),
            ]
        ], 200);
    }
}
