<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\AdminNotification;

class ParentLeaveRequestController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $requests = LeaveRequest::with(['student','leaveType'])
            ->where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->paginate(15);

        return view('parent.leave-requests.index', [
            'user' => $user,
            'requests' => $requests,
        ]);
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $leaveTypes = LeaveType::where('is_active', true)->orderBy('name')->get();
        $students = $user->students;

        return view('parent.leave-requests.create', [
            'user' => $user,
            'leaveTypes' => $leaveTypes,
            'students' => $students,
        ]);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'student_id' => 'required|exists:students,id',
            'leave_type_id' => 'nullable|exists:leave_types,id',
            'from_date' => 'required|date',
            'to_date' => 'required|date|after_or_equal:from_date',
            'reason' => 'nullable|string',
        ]);

        // Ensure the student belongs to this parent
        if (!$user->students()->where('id', $validated['student_id'])->exists()) {
            abort(403, 'You cannot request leave for this student.');
        }

        $leaveRequest = LeaveRequest::create([
            'user_id' => $user->id,
            'role' => 'parent',
            'student_id' => $validated['student_id'],
            'leave_type_id' => $validated['leave_type_id'] ?? null,
            'from_date' => $validated['from_date'],
            'to_date' => $validated['to_date'],
            'reason' => $validated['reason'] ?? null,
            'status' => 'pending',
        ]);

        // Send admin notification
        try {
            $leaveRequest->load(['student', 'leaveType']);
            AdminNotification::notifyLeaveRequest($leaveRequest, $user, 'Parent');
        } catch (\Exception $e) {
            \Log::error('Failed to create admin notification: ' . $e->getMessage());
        }

        return redirect()->route('parent.leave-requests.index')->with('success', 'Leave request submitted.');
    }
}


