<?php

namespace App\Http\Controllers;

use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ParentNotificationController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $childIds = $user->students->pluck('id')->toArray();
        $classIds = $user->students->pluck('class_id')->filter()->unique()->toArray();

        $notifications = Notification::where('is_active', true)
            ->where(function($q) use ($user, $childIds, $classIds) {
                $q->where('for_all_parents', true)
                  ->orWhere('parent_id', $user->id)
                  ->orWhereIn('student_id', $childIds)
                  ->orWhereIn('class_id', $classIds);
            })
            ->whereDoesntHave('reads', function($q) use ($user) {
                $q->where('user_id', $user->id)->whereNotNull('deleted_at');
            })
            ->orderByDesc('created_at')
            ->paginate(20);

        return view('parent.notifications.index', [
            'user' => $user,
            'notifications' => $notifications,
        ]);
    }

    public function markAsRead($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);
        $notification->markAsReadBy($user->id);

        if (request()->ajax()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Notification marked as read');
    }

    public function markAllAsRead()
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $childIds = $user->students->pluck('id')->toArray();
        $classIds = $user->students->pluck('class_id')->filter()->unique()->toArray();

        $notifications = Notification::where('is_active', true)
            ->where(function($q) use ($user, $childIds, $classIds) {
                $q->where('for_all_parents', true)
                  ->orWhere('parent_id', $user->id)
                  ->orWhereIn('student_id', $childIds)
                  ->orWhereIn('class_id', $classIds);
            })
            ->get();

        foreach ($notifications as $notification) {
            $notification->markAsReadBy($user->id);
        }

        if (request()->ajax()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'All notifications marked as read');
    }

    public function destroy($id)
    {
        $user = Auth::user();
        if (!$user || !$user->isParent()) {
            abort(403, 'Unauthorized access');
        }

        $notification = Notification::findOrFail($id);
        
        // Mark as deleted (soft delete for this user)
        \DB::table('notification_reads')
            ->updateOrInsert(
                ['notification_id' => $notification->id, 'user_id' => $user->id],
                ['read_at' => now(), 'deleted_at' => now()]
            );

        if (request()->ajax()) {
            return response()->json(['success' => true]);
        }

        return back()->with('success', 'Notification deleted');
    }
}
