<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Teacher;
use App\Models\Timetable;
use App\Models\TimetableOverride;
use App\Models\LeaveRequest;
use Carbon\Carbon;

class TeacherDashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::with('subjects')
            ->where('user_id', $user->id)
            ->first();

        $subjectCount = $teacher?->subjects?->count() ?? 0;

        $today = Carbon::today();
        $dayKey = strtolower($today->format('l'));
        $timeNow = Carbon::now()->format('H:i:s');

        $todayBase = $teacher ? Timetable::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->where('day_of_week', $dayKey)
            ->orderBy('start_time')
            ->get() : collect();

        $todayOverrides = $teacher ? TimetableOverride::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->whereDate('date', $today)
            ->orderBy('start_time')
            ->get() : collect();

        $todayClassCount = $todayBase->count() + $todayOverrides->count();

        $pendingLeaves = LeaveRequest::where('user_id', $user->id)
            ->where('status', 'pending')
            ->count();

        $currentOverride = $teacher ? TimetableOverride::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->whereDate('date', $today)
            ->where('start_time', '<=', $timeNow)
            ->where('end_time', '>', $timeNow)
            ->orderBy('start_time')
            ->first() : null;

        $currentBase = (!$currentOverride && $teacher) ? Timetable::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->where('day_of_week', $dayKey)
            ->where('start_time', '<=', $timeNow)
            ->where('end_time', '>', $timeNow)
            ->orderBy('start_time')
            ->first() : null;

        $nextOverride = $teacher ? TimetableOverride::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->whereDate('date', $today)
            ->where('start_time', '>', $timeNow)
            ->orderBy('start_time')
            ->first() : null;

        $nextBase = $teacher ? Timetable::with(['schoolClass','section','subject'])
            ->where('teacher_id', $teacher->id)
            ->where('day_of_week', $dayKey)
            ->where('start_time', '>', $timeNow)
            ->orderBy('start_time')
            ->first() : null;

        $nextSlot = null;
        if ($nextOverride && $nextBase) {
            $nextSlot = $nextOverride->start_time <= $nextBase->start_time ? $nextOverride : $nextBase;
        } else {
            $nextSlot = $nextOverride ?? $nextBase;
        }

        $currentSlot = $currentOverride ?? $currentBase;

        return view('teacher.dashboard', [
            'user' => $user,
            'teacher' => $teacher,
            'subjectCount' => $subjectCount,
            'todayClassCount' => $todayClassCount,
            'pendingLeaves' => $pendingLeaves,
            'currentSlot' => $currentSlot,
            'nextSlot' => $nextSlot,
        ]);
    }

    /**
     * Show the logged-in teacher their timetable.
     */
    public function timetable()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        // Base weekly timetable slots for this teacher
        $slots = Timetable::with(['schoolClass', 'section', 'subject'])
            ->where('teacher_id', $teacher->id)
            ->orderBy('day_of_week')
            ->orderBy('start_time')
            ->get();

        // Today + any per-date overrides for this teacher
        $today = Carbon::today();
        $overrides = TimetableOverride::with(['schoolClass', 'section', 'subject'])
            ->where('teacher_id', $teacher->id)
            ->whereDate('date', '>=', $today->copy()->subWeek())
            ->orderBy('date')
            ->orderBy('start_time')
            ->get();

        return view('teacher.timetable.index', [
            'user' => $user,
            'teacher' => $teacher,
            'slots' => $slots,
            'overrides' => $overrides,
        ]);
    }
}


