<?php

namespace App\Http\Controllers;

use App\Models\ExamSchedule;
use App\Models\ExamType;
use App\Models\ExamResult;
use App\Models\Room;
use App\Models\SchoolClass;
use App\Models\Section;
use App\Models\Student;
use App\Models\Subject;
use App\Models\Teacher;
use App\Services\PushNotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class TeacherExamController extends Controller
{
    public function index(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        $examTypes = ExamType::orderBy('name')->get();
        $classes = SchoolClass::orderBy('name')->get();
        $sections = Section::orderBy('name')->get();

        $query = ExamSchedule::with(['examType', 'schoolClass', 'section', 'subject', 'room'])
            ->where('teacher_id', $teacher->id)
            ->orderBy('exam_date')
            ->orderBy('start_time');

        if ($request->filled('exam_type_id')) {
            $query->where('exam_type_id', $request->exam_type_id);
        }
        if ($request->filled('class_id')) {
            $query->where('class_id', $request->class_id);
        }
        if ($request->filled('section_id')) {
            $query->where('section_id', $request->section_id);
        }

        $schedules = $query->get();

        return view('teacher.exams.schedules.index', [
            'user' => $user,
            'teacher' => $teacher,
            'examTypes' => $examTypes,
            'classes' => $classes,
            'sections' => $sections,
            'schedules' => $schedules,
            'filters' => $request->only(['exam_type_id', 'class_id', 'section_id']),
        ]);
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::with('subjects')->where('user_id', $user->id)->firstOrFail();

        return view('teacher.exams.schedules.create', [
            'user' => $user,
            'teacher' => $teacher,
            'examTypes' => ExamType::orderBy('name')->get(),
            'classes' => SchoolClass::orderBy('name')->get(),
            'sections' => Section::orderBy('name')->get(),
            'subjects' => $teacher->subjects->sortBy('name'),
            'rooms' => Room::where('is_active', true)->orderBy('name')->get(),
        ]);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        $validated = $request->validate([
            'exam_type_id' => 'required|exists:exam_types,id',
            'class_id' => 'required|exists:school_classes,id',
            'section_id' => 'required|exists:sections,id',
            'subject_id' => 'required|exists:subjects,id',
            'exam_date' => 'required|date',
            'start_time' => 'required|date_format:H:i',
            'end_time' => 'required|date_format:H:i|after:start_time',
            'room_id' => 'nullable|exists:rooms,id',
            'max_marks' => 'nullable|integer|min:1',
            'pass_mark' => 'nullable|integer|min:0',
            'note' => 'nullable|string',
        ]);

        $validated['teacher_id'] = $teacher->id;

        $examSchedule = ExamSchedule::create($validated);

        // Send notification to parents about exam schedule
        try {
            $examSchedule->load(['examType', 'subject', 'schoolClass']);
            $pushService = new PushNotificationService();
            $pushService->notifyExamScheduled($examSchedule);
        } catch (\Exception $e) {
            \Log::error('Failed to send exam schedule notification: ' . $e->getMessage());
        }

        return redirect()->route('teacher.exams.schedules.index')
            ->with('success', 'Exam schedule created successfully.');
    }

    public function marks(ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        abort_unless($examSchedule->teacher_id === $teacher->id, 403);

        $students = Student::where('class_id', $examSchedule->class_id)
            ->where('section_id', $examSchedule->section_id)
            ->orderBy('first_name')
            ->get();

        $existingResults = $examSchedule->results()
            ->get()
            ->keyBy('student_id');

        return view('teacher.exams.schedules.marks', [
            'user' => $user,
            'teacher' => $teacher,
            'examSchedule' => $examSchedule->load(['examType', 'schoolClass', 'section', 'subject']),
            'students' => $students,
            'results' => $existingResults,
        ]);
    }

    public function storeMarks(Request $request, ExamSchedule $examSchedule)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        abort_unless($examSchedule->teacher_id === $teacher->id, 403);

        $data = $request->validate([
            'marks' => 'array',
            'marks.*' => 'nullable|integer|min:0',
            'absent' => 'array',
            'absent.*' => 'nullable|boolean',
            'remark' => 'array',
            'remark.*' => 'nullable|string',
        ]);

        $maxMarks = $examSchedule->max_marks;
        $passMark = $examSchedule->pass_mark;

        foreach ($data['marks'] ?? [] as $studentId => $marks) {
            $isAbsent = isset($data['absent'][$studentId]) && $data['absent'][$studentId];
            $remark = $data['remark'][$studentId] ?? null;

            if ($isAbsent) {
                $marks = null;
            } elseif ($marks !== null && $maxMarks !== null && $marks > $maxMarks) {
                $marks = $maxMarks;
            }

            $grade = null;
            if (!$isAbsent && $marks !== null && $maxMarks !== null && $maxMarks > 0) {
                $percentage = ($marks / $maxMarks) * 100;
                if ($percentage >= 90) {
                    $grade = 'A+';
                } elseif ($percentage >= 80) {
                    $grade = 'A';
                } elseif ($percentage >= 70) {
                    $grade = 'B+';
                } elseif ($percentage >= 60) {
                    $grade = 'B';
                } elseif ($percentage >= 50) {
                    $grade = 'C';
                } else {
                    $grade = 'D';
                }
            }

            ExamResult::updateOrCreate(
                [
                    'exam_schedule_id' => $examSchedule->id,
                    'student_id' => $studentId,
                ],
                [
                    'marks_obtained' => $marks,
                    'is_absent' => $isAbsent,
                    'grade' => $grade,
                    'remark' => $remark,
                ]
            );
        }

        // Send notifications to parents about results
        try {
            $examSchedule->load(['examType', 'subject']);
            $pushService = new PushNotificationService();
            
            // Get all students who got results
            $studentIds = array_keys($data['marks'] ?? []);
            $students = Student::whereIn('id', $studentIds)->whereNotNull('parent_id')->get();
            
            foreach ($students as $student) {
                $result = ExamResult::where('exam_schedule_id', $examSchedule->id)
                    ->where('student_id', $student->id)
                    ->first();
                
                if ($result && !$result->is_absent) {
                    $pushService->notifyExamResult($examSchedule, $student, $result);
                }
            }
        } catch (\Exception $e) {
            \Log::error('Failed to send exam result notifications: ' . $e->getMessage());
        }

        return redirect()
            ->route('teacher.exams.schedules.index')
            ->with('success', 'Marks saved successfully.');
    }
}


