<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use App\Models\AdminNotification;

class TeacherLeaveRequestController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $requests = LeaveRequest::with(['leaveType'])
            ->where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->paginate(15);

        return view('teacher.leave-requests.index', [
            'user' => $user,
            'requests' => $requests,
        ]);
    }

    public function create()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $leaveTypes = LeaveType::where('is_active', true)->orderBy('name')->get();

        return view('teacher.leave-requests.create', [
            'user' => $user,
            'leaveTypes' => $leaveTypes,
        ]);
    }

    public function store(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $validated = $request->validate([
            'leave_type_id' => 'nullable|exists:leave_types,id',
            'from_date' => 'required|date',
            'to_date' => 'required|date|after_or_equal:from_date',
            'reason' => 'nullable|string',
        ]);

        $leaveRequest = LeaveRequest::create([
            'user_id' => $user->id,
            'role' => 'teacher',
            'student_id' => null,
            'leave_type_id' => $validated['leave_type_id'] ?? null,
            'from_date' => $validated['from_date'],
            'to_date' => $validated['to_date'],
            'reason' => $validated['reason'] ?? null,
            'status' => 'pending',
        ]);

        // Send admin notification
        try {
            $leaveRequest->load('leaveType');
            $leaveTypeName = $leaveRequest->leaveType->name ?? 'Leave';
            $fromDate = \Carbon\Carbon::parse($validated['from_date'])->format('d M');
            $toDate = \Carbon\Carbon::parse($validated['to_date'])->format('d M');
            
            AdminNotification::create([
                'type' => 'leave_request',
                'title' => 'Teacher Leave Request',
                'message' => "{$user->name} (Teacher) requested {$leaveTypeName} from {$fromDate} to {$toDate}",
                'icon' => 'ri-calendar-check-line',
                'icon_bg' => 'bg-info-600',
                'link' => route('admin.leave-requests.index'),
                'from_user_id' => $user->id,
                'from_role' => 'teacher',
                'data' => [
                    'leave_request_id' => $leaveRequest->id,
                    'teacher_name' => $user->name,
                ],
            ]);
        } catch (\Exception $e) {
            \Log::error('Failed to create admin notification: ' . $e->getMessage());
        }

        return redirect()->route('teacher.leave-requests.index')->with('success', 'Leave request submitted.');
    }
}


