<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Models\Teacher;

class TeacherProfileController extends Controller
{
    public function edit()
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        return view('teacher.profile.edit', [
            'user' => $user,
            'teacher' => $teacher,
        ]);
    }

    public function update(Request $request)
    {
        $user = Auth::user();
        if (!$user || !$user->isTeacher()) {
            abort(403, 'Unauthorized access');
        }

        $teacher = Teacher::where('user_id', $user->id)->firstOrFail();

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|string|max:20',
            'qualification' => 'nullable|string|max:255',
            'experience' => 'nullable|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        // Update teacher
        $teacher->fill([
            'name' => $validated['name'],
            'email' => $validated['email'] ?? $teacher->email,
            'phone' => $validated['phone'] ?? $teacher->phone,
            'date_of_birth' => $validated['date_of_birth'] ?? $teacher->date_of_birth,
            'gender' => $validated['gender'] ?? $teacher->gender,
            'qualification' => $validated['qualification'] ?? $teacher->qualification,
            'experience' => $validated['experience'] ?? $teacher->experience,
        ]);

        if ($request->hasFile('photo')) {
            if ($teacher->photo) {
                Storage::disk('public')->delete($teacher->photo);
            }
            $teacher->photo = $request->file('photo')->store('teachers', 'public');
        }

        $teacher->save();

        // Optionally sync name to user record for display
        $user->name = $teacher->name;
        $user->save();

        return redirect()->route('teacher.profile.edit')->with('success', 'Profile updated successfully.');
    }
}


