@extends('layouts.admin')

@section('title', 'Add Student - Admission')

@section('content')
<div class="breadcrumb d-flex flex-wrap align-items-center justify-content-between gap-3 mb-24">
    <div class="">
        <h1 class="fw-semibold mb-4 h6 text-primary-light">Add New Student</h1>
        <div class="">
            <a href="{{ route('admin.dashboard') }}" class="text-secondary-light hover-text-primary hover-underline">Dashboard</a>
            <a href="{{ route('admin.student.index') }}" class="text-secondary-light hover-text-primary hover-underline">/ Student</a>
            <span class="text-secondary-light">/ Add New Student</span>
        </div>

    </div>
</div>

@if(isset($errors) && $errors->any())
<div class="alert alert-danger">
    <ul class="mb-0">
        @foreach ($errors->all() as $error)
        <li>{{ $error }}</li>
        @endforeach
    </ul>
</div>
@endif

<form action="{{ route('admin.student.store') }}" method="POST" enctype="multipart/form-data" class="mt-24">
    @csrf
    <div class="row gy-3">
        <!-- Basic Information -->
        <div class="col-lg-12">
            <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
                <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
                    <h6 class="text-lg fw-semibold mb-0">Basic Information</h6>
                </div>
                <div class="card-body p-20">
                    <div class="row gy-3">
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="academic_year_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Academic Year <span class="text-danger-600">*</span>
                                </label>
                                <select id="academic_year_id" name="academic_year_id" class="form-control form-select @error('academic_year_id') is-invalid @enderror" required>
                                    <option value="">Select Academic Year</option>
                                    @foreach($academicYears as $year)
                                    <option value="{{ $year->id }}" {{ old('academic_year_id') == $year->id ? 'selected' : '' }}>
                                        {{ $year->name }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('academic_year_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="class_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Class <span class="text-danger-600">*</span>
                                </label>
                                <select id="class_id" name="class_id" class="form-control form-select @error('class_id') is-invalid @enderror" required>
                                    <option value="">Select Class</option>
                                    @foreach($classes as $class)
                                    <option value="{{ $class->id }}" {{ old('class_id') == $class->id ? 'selected' : '' }}>
                                        {{ $class->name }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('class_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="section_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Section <span class="text-danger-600">*</span>
                                </label>
                                <select id="section_id" name="section_id" class="form-control form-select @error('section_id') is-invalid @enderror" required>
                                    <option value="">Select Section</option>
                                    @foreach($sections as $section)
                                    @php
                                        $filledSeats = $section->students_count ?? 0;
                                        $totalSeats = $section->seats ?? 20;
                                        $availableSeats = max(0, $totalSeats - $filledSeats);
                                    @endphp
                                    <option value="{{ $section->id }}" {{ old('section_id') == $section->id ? 'selected' : '' }}>
                                        {{ $section->name }} ({{ $availableSeats }} seats available)
                                    </option>
                                    @endforeach
                                </select>
                                <small class="text-secondary-light">Only sections with available seats are shown</small>
                                @error('section_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="batch_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Batch
                                </label>
                                <select id="batch_id" name="batch_id" class="form-control form-select">
                                    <option value="">Select Batch</option>
                                    @foreach($batches as $batch)
                                    <option value="{{ $batch->id }}" {{ old('batch_id') == $batch->id ? 'selected' : '' }}>
                                        {{ $batch->name }}
                                    </option>
                                    @endforeach
                                </select>
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Admission No
                                </label>
                                <input type="text" class="form-control bg-light" 
                                    value="Auto-generated (e.g., ADM-2026-0001)" readonly disabled>
                                <small class="text-secondary-light">Admission number will be auto-generated when you save</small>
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="name" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Student Name <span class="text-danger-600">*</span>
                                </label>
                                <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                    id="name" name="name" placeholder="Enter Student Name" 
                                    value="{{ old('name') }}" required>
                                @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="gender" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Gender <span class="text-danger-600">*</span>
                                </label>
                                <select id="gender" name="gender" class="form-control form-select @error('gender') is-invalid @enderror" required>
                                    <option value="">Select Gender</option>
                                    <option value="Male" {{ old('gender') == 'Male' ? 'selected' : '' }}>Male</option>
                                    <option value="Female" {{ old('gender') == 'Female' ? 'selected' : '' }}>Female</option>
                                    <option value="Other" {{ old('gender') == 'Other' ? 'selected' : '' }}>Other</option>
                                </select>
                                @error('gender')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="date_of_birth" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Date Of Birth <span class="text-danger-600">*</span>
                                </label>
                                <input type="date" class="form-control @error('date_of_birth') is-invalid @enderror" 
                                    id="date_of_birth" name="date_of_birth" value="{{ old('date_of_birth') }}" required>
                                @error('date_of_birth')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="email" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Email <span class="text-danger-600">*</span>
                                </label>
                                <input type="email" class="form-control @error('email') is-invalid @enderror" 
                                    id="email" name="email" placeholder="Enter Email" 
                                    value="{{ old('email') }}" required>
                                @error('email')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="photo" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Student Photo <span class="text-danger-600">*</span>
                                </label>
                                <input type="file" class="form-control @error('photo') is-invalid @enderror" 
                                    id="photo" name="photo" accept="image/*" required>
                                @error('photo')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Current School Details -->
        <div class="col-lg-12">
            <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
                <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
                    <h6 class="text-lg fw-semibold mb-0">Current School Details</h6>
                </div>
                <div class="card-body p-20">
                    <div class="row gy-3">
                        <div class="col-xxl-6 col-xl-6 col-sm-12">
                            <div class="">
                                <label for="previous_school_name" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">School Name</label>
                                <input type="text" class="form-control" id="previous_school_name" name="previous_school_name" placeholder="Enter Current School Name" value="{{ old('previous_school_name') }}">
                            </div>
                        </div>
                        <div class="col-xxl-6 col-xl-6 col-sm-12">
                            <div class="">
                                <label for="previous_school_address" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">School Address</label>
                                <input type="text" class="form-control" id="previous_school_address" name="previous_school_address" placeholder="Enter School Address" value="{{ old('previous_school_address') }}">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Parent Details -->
        <div class="col-lg-12">
            <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
                <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
                    <h6 class="text-lg fw-semibold mb-0">Parent Details</h6>
                </div>
                <div class="card-body p-20">
                    <div class="row gy-3">
                        <div class="col-12 mb-3">
                            <div class="alert alert-info">
                                <strong>Note:</strong> Search for an existing parent if they already have a child registered. Otherwise, provide parent details to create a new parent account.
                            </div>
                        </div>
                        <div class="col-12 mb-3">
                            <label class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                Search Existing Parent (Optional)
                            </label>
                            <div class="position-relative">
                                <input type="text" 
                                    id="parent_search" 
                                    class="form-control" 
                                    placeholder="Search by name, email, or phone..."
                                    autocomplete="off"
                                    value="{{ old('existing_parent_id') ? '' : '' }}">
                                <input type="hidden" id="existing_parent_id" name="existing_parent_id" value="{{ old('existing_parent_id') }}">
                                <div id="parent_search_results" class="position-absolute w-100 bg-white border shadow-lg radius-8 mt-1" style="display: none; z-index: 1050; max-height: 300px; overflow-y: auto; left: 0; right: 0; box-shadow: 0 4px 6px rgba(0,0,0,0.1);"></div>
                            </div>
                            <small class="text-secondary-light">If parent already exists, select them to avoid creating duplicate accounts</small>
                        </div>
                        <div id="selected_parent_info" class="col-12 mb-3" style="display: none;">
                            <div class="alert alert-success">
                                <strong>Selected Parent:</strong> <span id="selected_parent_name"></span> 
                                (<span id="selected_parent_email"></span>) - 
                                <span id="selected_parent_students"></span> child(ren) already registered
                                <button type="button" class="btn btn-sm btn-outline-danger ms-2" onclick="clearParentSelection()">Clear Selection</button>
                            </div>
                        </div>
                        <div id="new_parent_fields">
                            <div class="row gy-3">
                                <div class="col-xxl-4 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_name" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Name <span class="text-danger-600 required-asterisk">*</span>
                                        </label>
                                        <input type="text" class="form-control @error('parent_name') is-invalid @enderror" 
                                            id="parent_name" name="parent_name" 
                                            placeholder="Enter Parent Name" 
                                            value="{{ old('parent_name') }}" required>
                                        @error('parent_name')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                    </div>
                                </div>
                                <div class="col-xxl-4 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_email" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Email <span class="text-danger-600 required-asterisk">*</span> (for login)
                                        </label>
                                        <input type="email" class="form-control @error('parent_email') is-invalid @enderror" 
                                            id="parent_email" name="parent_email" 
                                            placeholder="Enter Parent Email" 
                                            value="{{ old('parent_email') }}" required>
                                        @error('parent_email')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-secondary-light">This will be used as login email for parent</small>
                                    </div>
                                </div>
                                <div class="col-xxl-4 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_password" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Password <span class="text-danger-600 required-asterisk">*</span> (for login)
                                        </label>
                                        <div class="position-relative">
                                            <input type="password" class="form-control @error('parent_password') is-invalid @enderror" 
                                                id="parent_password" name="parent_password" 
                                                placeholder="Enter Password (min 6 characters)" 
                                                minlength="6" required>
                                            <span class="toggle-password ri-eye-line cursor-pointer position-absolute end-0 top-50 translate-middle-y me-16 text-secondary-light"
                                                data-toggle="#parent_password"></span>
                                        </div>
                                        @error('parent_password')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                        @enderror
                                        <small class="text-secondary-light">Minimum 6 characters required</small>
                                    </div>
                                </div>
                                <div class="col-xxl-3 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_phone" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Phone Number
                                        </label>
                                        <input type="text" class="form-control" 
                                            id="parent_phone" name="parent_phone" 
                                            placeholder="Enter Phone Number" 
                                            value="{{ old('parent_phone') }}">
                                    </div>
                                </div>
                                <div class="col-xxl-3 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_occupation" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Occupation
                                        </label>
                                        <input type="text" class="form-control" 
                                            id="parent_occupation" name="parent_occupation" 
                                            placeholder="Enter Occupation" 
                                            value="{{ old('parent_occupation') }}">
                                    </div>
                                </div>
                                <div class="col-xxl-3 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_address" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Address
                                        </label>
                                        <input type="text" class="form-control" 
                                            id="parent_address" name="parent_address" 
                                            placeholder="Enter Address" 
                                            value="{{ old('parent_address') }}">
                                    </div>
                                </div>
                                <div class="col-xxl-3 col-xl-4 col-md-6 col-sm-12">
                                    <div class="">
                                        <label for="parent_photo" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                            Parent Photo
                                        </label>
                                        <input type="file" class="form-control" 
                                            id="parent_photo" name="parent_photo" 
                                            accept="image/*">
                                    </div>
                                </div>
                            </div>
                        </div> <!-- End new_parent_fields -->
                    </div>
                </div>
            </div>
        </div>

        <!-- Medical Details -->
        <div class="col-lg-12">
            <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
                <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
                    <h6 class="text-lg fw-semibold mb-0">Medical Details</h6>
                </div>
                <div class="card-body p-20">
                    <div class="row gy-3">
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="blood_group" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">Blood Group</label>
                                <select id="blood_group" name="blood_group" class="form-control form-select">
                                    <option value="">Select blood group</option>
                                    <option value="A+" {{ old('blood_group') == 'A+' ? 'selected' : '' }}>A+</option>
                                    <option value="A-" {{ old('blood_group') == 'A-' ? 'selected' : '' }}>A-</option>
                                    <option value="B+" {{ old('blood_group') == 'B+' ? 'selected' : '' }}>B+</option>
                                    <option value="B-" {{ old('blood_group') == 'B-' ? 'selected' : '' }}>B-</option>
                                    <option value="AB+" {{ old('blood_group') == 'AB+' ? 'selected' : '' }}>AB+</option>
                                    <option value="AB-" {{ old('blood_group') == 'AB-' ? 'selected' : '' }}>AB-</option>
                                    <option value="O+" {{ old('blood_group') == 'O+' ? 'selected' : '' }}>O+</option>
                                    <option value="O-" {{ old('blood_group') == 'O-' ? 'selected' : '' }}>O-</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="height" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">Height</label>
                                <input type="text" class="form-control" id="height" name="height" placeholder="Enter height (e.g., 5'6\" or 170 cm)" value="{{ old('height') }}">
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="weight" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">Weight</label>
                                <input type="text" class="form-control" id="weight" name="weight" placeholder="Enter weight (e.g., 65 kg)" value="{{ old('weight') }}">
                            </div>
                        </div>
                        <div class="col-12">
                            <div class="mt-24">
                                <span class="text-lg fw-bold text-primary-light d-inline-block mb-8">Does the student need any medical care?</span>
                                <div class="d-flex align-items-center flex-wrap gap-28 mt-3">
                                    <div class="form-check checked-primary d-flex align-items-center gap-2">
                                        <input class="form-check-input" type="radio" name="medical_care_needed" id="medical_care_yes" value="yes" {{ old('medical_care_needed') == 'yes' ? 'checked' : '' }}>
                                        <label class="form-check-label line-height-1 fw-medium text-secondary-light" for="medical_care_yes">Yes</label>
                                    </div>
                                    <div class="form-check checked-secondary d-flex align-items-center gap-2">
                                        <input class="form-check-input" type="radio" name="medical_care_needed" id="medical_care_no" value="no" {{ old('medical_care_needed') == 'no' || old('medical_care_needed') == null ? 'checked' : '' }}>
                                        <label class="form-check-label line-height-1 fw-medium text-secondary-light" for="medical_care_no">No</label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-12" id="medical_care_details_section" style="display: {{ old('medical_care_needed') == 'yes' ? 'block' : 'none' }};">
                            <div class="">
                                <label for="medical_care_details" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">Medical Care Details <span class="text-danger-600">*</span></label>
                                <textarea id="medical_care_details" name="medical_care_details" class="form-control @error('medical_care_details') is-invalid @enderror" rows="4" placeholder="Enter medical care details (medications, conditions, special requirements, etc.)">{{ old('medical_care_details') }}</textarea>
                                @error('medical_care_details')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-12">
            <div class="d-flex align-items-center justify-content-center gap-3 mt-8">
                <a href="{{ route('admin.student.index') }}" 
                    class="border border-danger-600 bg-hover-danger-200 text-danger-600 text-md px-50 py-11 radius-8">
                    Cancel
                </a>
                <button type="submit"
                    class="btn btn-primary-600 border border-primary-600 text-md px-28 py-12 radius-8">
                    Save Changes
                </button>
            </div>
        </div>
    </div>
</form>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Toggle password visibility
        const togglePassword = document.querySelector('.toggle-password');
        if (togglePassword) {
            togglePassword.addEventListener('click', function() {
                const passwordInput = document.querySelector(this.getAttribute('data-toggle'));
                if (passwordInput) {
                    const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
                    passwordInput.setAttribute('type', type);
                    this.classList.toggle('ri-eye-line');
                    this.classList.toggle('ri-eye-off-line');
                }
            });
        }

        // Show/hide medical care details based on radio button selection
        const medicalCareYes = document.getElementById('medical_care_yes');
        const medicalCareNo = document.getElementById('medical_care_no');
        const medicalCareDetailsSection = document.getElementById('medical_care_details_section');
        const medicalCareDetails = document.getElementById('medical_care_details');

        function toggleMedicalCareDetails() {
            if (medicalCareYes && medicalCareYes.checked) {
                if (medicalCareDetailsSection) medicalCareDetailsSection.style.display = 'block';
                if (medicalCareDetails) medicalCareDetails.setAttribute('required', 'required');
            } else {
                if (medicalCareDetailsSection) medicalCareDetailsSection.style.display = 'none';
                if (medicalCareDetails) {
                    medicalCareDetails.removeAttribute('required');
                    medicalCareDetails.value = '';
                }
            }
        }

        if (medicalCareYes) {
            medicalCareYes.addEventListener('change', toggleMedicalCareDetails);
        }
        if (medicalCareNo) {
            medicalCareNo.addEventListener('change', toggleMedicalCareDetails);
        }
        
        // Initial check
        toggleMedicalCareDetails();

        // Parent search functionality
        const parentSearch = document.getElementById('parent_search');
        const parentSearchResults = document.getElementById('parent_search_results');
        const existingParentId = document.getElementById('existing_parent_id');
        const selectedParentInfo = document.getElementById('selected_parent_info');
        const newParentFields = document.getElementById('new_parent_fields');
        let searchTimeout;

        if (parentSearch) {
            parentSearch.addEventListener('input', function() {
                const query = this.value.trim();
                
                clearTimeout(searchTimeout);
                
                if (query.length < 1) {
                    parentSearchResults.style.display = 'none';
                    return;
                }

                searchTimeout = setTimeout(() => {
                    // Use full URL path to avoid route resolution issues
                    const baseUrl = window.location.origin;
                    // Ensure we're using the correct path
                    const searchUrl = `${baseUrl}/admin/student/search-parents?q=${encodeURIComponent(query)}`;
                    console.log('Search URL:', searchUrl);
                    console.log('Current location:', window.location.href);
                    
                    // Show loading
                    parentSearchResults.innerHTML = '<div class="p-2 text-secondary-light"><i class="ri-loader-4-line ri-spin"></i> Searching...</div>';
                    parentSearchResults.style.display = 'block';
                    
                    fetch(searchUrl, {
                        method: 'GET',
                        headers: {
                            'Accept': 'application/json',
                            'Content-Type': 'application/json',
                            'X-Requested-With': 'XMLHttpRequest',
                        },
                        credentials: 'same-origin'
                    })
                        .then(response => {
                            console.log('Response status:', response.status, 'URL:', searchUrl);
                            if (!response.ok) {
                                if (response.status === 404) {
                                    throw new Error(`Route not found. Please check if the route /admin/student/search-parents exists.`);
                                }
                                throw new Error(`HTTP error! status: ${response.status}`);
                            }
                            return response.json();
                        })
                        .then(data => {
                            if (data && Array.isArray(data) && data.length > 0) {
                                let html = '';
                                data.forEach(parent => {
                                    const escapedName = String(parent.name || '').replace(/'/g, "\\'").replace(/"/g, '&quot;').replace(/\n/g, ' ');
                                    const escapedEmail = String(parent.email || '').replace(/'/g, "\\'").replace(/"/g, '&quot;');
                                    const phone = parent.phone ? String(parent.phone) : '';
                                    const studentsCount = parent.students_count || 0;
                                    html += `<div class="p-3 border-bottom cursor-pointer" 
                                                onclick="selectParent(${parent.id}, '${escapedName}', '${escapedEmail}', ${studentsCount})"
                                                onmouseover="this.style.backgroundColor='#f8f9fa'" 
                                                onmouseout="this.style.backgroundColor='transparent'"
                                                style="transition: background-color 0.2s;">
                                                <div class="fw-semibold text-primary-light">${parent.name || 'N/A'}</div>
                                                <div class="text-sm text-secondary-light">${parent.email || 'N/A'}</div>
                                                ${phone ? `<div class="text-sm text-secondary-light mt-1">${phone}</div>` : ''}
                                                <div class="text-xs text-secondary-light mt-1">
                                                    <i class="ri-user-line"></i> ${studentsCount} child(ren) registered
                                                </div>
                                            </div>`;
                                });
                                parentSearchResults.innerHTML = html;
                                parentSearchResults.style.display = 'block';
                            } else {
                                parentSearchResults.innerHTML = '<div class="p-3 text-secondary-light text-center">No parents found matching "' + query + '"</div>';
                                parentSearchResults.style.display = 'block';
                            }
                        })
                        .catch(error => {
                            console.error('Search error:', error);
                            parentSearchResults.innerHTML = '<div class="p-3 text-danger text-center">Error searching parents. Please check console for details.</div>';
                            parentSearchResults.style.display = 'block';
                        });
                }, 300);
            });

            // Hide results when clicking outside
            document.addEventListener('click', function(e) {
                if (!parentSearch.contains(e.target) && !parentSearchResults.contains(e.target)) {
                    parentSearchResults.style.display = 'none';
                }
            });
        }

        window.selectParent = function(id, name, email, studentsCount) {
            existingParentId.value = id;
            document.getElementById('selected_parent_name').textContent = name;
            document.getElementById('selected_parent_email').textContent = email;
            document.getElementById('selected_parent_students').textContent = studentsCount;
            
            selectedParentInfo.style.display = 'block';
            newParentFields.style.display = 'none';
            parentSearch.value = name;
            parentSearchResults.style.display = 'none';
            
            // Remove required attributes from new parent fields
            const parentName = document.getElementById('parent_name');
            const parentEmail = document.getElementById('parent_email');
            const parentPassword = document.getElementById('parent_password');
            
            if (parentName) parentName.removeAttribute('required');
            if (parentEmail) parentEmail.removeAttribute('required');
            if (parentPassword) parentPassword.removeAttribute('required');
            
            // Hide required asterisks
            document.querySelectorAll('.required-asterisk').forEach(el => {
                el.style.display = 'none';
            });
        };

        window.clearParentSelection = function() {
            existingParentId.value = '';
            selectedParentInfo.style.display = 'none';
            newParentFields.style.display = 'block';
            parentSearch.value = '';
            
            // Add required attributes back to new parent fields
            const parentName = document.getElementById('parent_name');
            const parentEmail = document.getElementById('parent_email');
            const parentPassword = document.getElementById('parent_password');
            
            if (parentName) parentName.setAttribute('required', 'required');
            if (parentEmail) parentEmail.setAttribute('required', 'required');
            if (parentPassword) parentPassword.setAttribute('required', 'required');
            
            // Show required asterisks
            document.querySelectorAll('.required-asterisk').forEach(el => {
                el.style.display = 'inline';
            });
        };

        // Check if existing parent is already selected (from old input)
        @if(old('existing_parent_id'))
            const oldParentId = {{ old('existing_parent_id') }};
            const baseUrlForOld = window.location.origin;
            fetch(`${baseUrlForOld}/admin/student/search-parents?id=${oldParentId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.length > 0) {
                        const parent = data[0];
                        selectParent(parent.id, parent.name, parent.email, parent.students_count);
                    }
                });
        @endif

    });
</script>
@endsection

