@extends('layouts.admin')

@section('title', 'Add Timetable Slot')

@section('content')
<div class="breadcrumb d-flex flex-wrap align-items-center justify-content-between gap-3 mb-24">
    <div class="">
        <h1 class="fw-semibold mb-4 h6 text-primary-light">Add Timetable Slot</h1>
        <div class="">
            <a href="{{ route('admin.dashboard') }}" class="text-secondary-light hover-text-primary hover-underline">Dashboard</a>
            <a href="{{ route('admin.timetable.index') }}" class="text-secondary-light hover-text-primary hover-underline">/ Timetable</a>
            <span class="text-secondary-light">/ Add Slot</span>
        </div>
    </div>
</div>

@if(session('success'))
<div class="alert alert-success alert-dismissible fade show" role="alert">
    {{ session('success') }}
    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
</div>
@endif

@if(isset($errors) && $errors->any())
<div class="alert alert-danger">
    <ul class="mb-0">
        @foreach ($errors->all() as $error)
        <li>{{ $error }}</li>
        @endforeach
    </ul>
</div>
@endif

<form action="{{ route('admin.timetable.store') }}" method="POST" class="mt-24">
    @csrf
    <div class="row gy-3">
        <div class="col-lg-12">
            <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
                <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
                    <h6 class="text-lg fw-semibold mb-0">Timetable Slot</h6>
                </div>
                <div class="card-body p-20">
                    <div class="row gy-3">
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="class_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Class <span class="text-danger-600">*</span>
                                </label>
                                <select id="class_id" name="class_id" class="form-control form-select @error('class_id') is-invalid @enderror" required>
                                    <option value="">Select Class</option>
                                    @foreach($classes as $class)
                                    <option value="{{ $class->id }}" {{ old('class_id') == $class->id ? 'selected' : '' }}>
                                        {{ $class->name }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('class_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="section_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Section <span class="text-danger-600">*</span>
                                </label>
                                <select id="section_id" name="section_id" class="form-control form-select @error('section_id') is-invalid @enderror" required>
                                    <option value="">Select Section</option>
                                    @foreach($sections as $section)
                                    <option value="{{ $section->id }}" {{ old('section_id') == $section->id ? 'selected' : '' }}>
                                        {{ $section->name }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('section_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="day_of_week" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Day <span class="text-danger-600">*</span>
                                </label>
                                <select id="day_of_week" name="day_of_week" class="form-control form-select @error('day_of_week') is-invalid @enderror" required>
                                    <option value="">Select Day</option>
                                    @foreach($days as $d)
                                    <option value="{{ $d }}" {{ old('day_of_week') == $d ? 'selected' : '' }}>
                                        {{ ucfirst($d) }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('day_of_week')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="subject_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Subject <span class="text-danger-600">*</span>
                                </label>
                                <select id="subject_id" name="subject_id" class="form-control form-select @error('subject_id') is-invalid @enderror" required>
                                    <option value="">Select Subject</option>
                                    @foreach($subjects as $subject)
                                    <option value="{{ $subject->id }}" {{ old('subject_id') == $subject->id ? 'selected' : '' }}>
                                        {{ $subject->name }}
                                    </option>
                                    @endforeach
                                </select>
                                @error('subject_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="teacher_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Teacher <span class="text-danger-600">*</span>
                                </label>
                                <select id="teacher_id" name="teacher_id" class="form-control form-select @error('teacher_id') is-invalid @enderror" required>
                                    <option value="">Select Teacher</option>
                                    @foreach($teachers as $teacher)
                                    <option value="{{ $teacher->id }}"
                                        data-subjects="{{ $teacher->subjects->pluck('id')->join(',') }}"
                                        {{ old('teacher_id') == $teacher->id ? 'selected' : '' }}>
                                        {{ $teacher->name }}
                                    </option>
                                    @endforeach
                                </select>
                                <small class="text-secondary-light d-block">Teachers filtered by selected subject</small>
                                @error('teacher_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                @error('teacher_id_conflict')
                                <div class="invalid-feedback d-block">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="start_time" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Start Time <span class="text-danger-600">*</span>
                                </label>
                                <input type="time" class="form-control @error('start_time') is-invalid @enderror" 
                                    id="start_time" name="start_time" value="{{ old('start_time') }}" required>
                                @error('start_time')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-secondary-light">Format: 07:00 AM</small>
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="end_time" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    End Time <span class="text-danger-600">*</span>
                                </label>
                                <input type="time" class="form-control @error('end_time') is-invalid @enderror" 
                                    id="end_time" name="end_time" value="{{ old('end_time') }}" required>
                                @error('end_time')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                                <small class="text-secondary-light">Format: 08:00 AM</small>
                            </div>
                        </div>
                        <div class="col-xxl-3 col-xl-4 col-sm-6">
                            <div class="">
                                <label for="room_id" class="text-sm fw-semibold text-primary-light d-inline-block mb-8">
                                    Room
                                </label>
                                <select id="room_id" name="room_id" class="form-control form-select @error('room_id') is-invalid @enderror">
                                    <option value="">Select Room</option>
                                    @foreach($rooms as $room)
                                    <option value="{{ $room->id }}" {{ old('room_id') == $room->id ? 'selected' : '' }}>
                                        {{ $room->name }}@if($room->capacity) ({{ $room->capacity }})@endif
                                    </option>
                                    @endforeach
                                </select>
                                @error('room_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-12">
            <div class="d-flex align-items-center justify-content-center gap-3 mt-8">
                <a href="{{ route('admin.timetable.index') }}" 
                    class="border border-danger-600 bg-hover-danger-200 text-danger-600 text-md px-50 py-11 radius-8">
                    Cancel
                </a>
                <button type="submit"
                    class="btn btn-primary-600 border border-primary-600 text-md px-28 py-12 radius-8">
                    Save Slot
                </button>
            </div>
        </div>
    </div>
</form>

@if(isset($timetables) && $timetables->count() > 0)
<div class="mt-24">
    <div class="shadow-1 radius-12 bg-base h-100 overflow-hidden">
        <div class="card-header border-bottom bg-base py-16 px-24 d-flex align-items-center justify-content-between">
            <h6 class="text-lg fw-semibold mb-0">Existing Timetable Slots</h6>
        </div>
        <div class="card-body p-20">
            <div class="table-responsive">
                <table class="table bordered-table mb-0">
                    <thead>
                        <tr>
                            <th scope="col" style="width: 50px;">S.L</th>
                            <th scope="col">Time</th>
                            <th scope="col">Day</th>
                            <th scope="col">Class</th>
                            <th scope="col">Section</th>
                            <th scope="col">Subject</th>
                            <th scope="col">Teacher</th>
                            <th scope="col">Room</th>
                            <th scope="col" style="width: 100px;">Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        @foreach($timetables as $index => $slot)
                        <tr>
                            <td>{{ $index + 1 }}</td>
                            <td>
                                {{ \Carbon\Carbon::parse($slot->start_time)->format('h:i A') }}
                                -
                                {{ \Carbon\Carbon::parse($slot->end_time)->format('h:i A') }}
                            </td>
                            <td>{{ ucfirst($slot->day_of_week) }}</td>
                            <td>{{ $slot->schoolClass ? $slot->schoolClass->name : 'N/A' }}</td>
                            <td>{{ $slot->section ? $slot->section->name : 'N/A' }}</td>
                            <td>{{ $slot->subject ? $slot->subject->name : 'N/A' }}</td>
                            <td>{{ $slot->teacher ? $slot->teacher->name : 'N/A' }}</td>
                            <td>{{ $slot->room ?? 'N/A' }}</td>
                            <td>
                                <div class="d-flex flex-wrap gap-2">
                                    <a href="{{ route('admin.timetable.edit', $slot->id) }}" 
                                        class="btn btn-sm btn-primary-600 d-flex align-items-center justify-content-center" 
                                        style="width: 32px; height: 32px; padding: 0;"
                                        title="Edit">
                                        <i class="ri-pencil-line"></i>
                                    </a>
                                    <form action="{{ route('admin.timetable.destroy', $slot->id) }}" method="POST" class="d-inline">
                                        @csrf
                                        @method('DELETE')
                                        <button type="submit" 
                                            class="btn btn-sm btn-danger-600 d-flex align-items-center justify-content-center"
                                            style="width: 32px; height: 32px; padding: 0;"
                                            onclick="return confirm('Are you sure you want to delete this slot?');"
                                            title="Delete">
                                            <i class="ri-delete-bin-line"></i>
                                        </button>
                                    </form>
                                </div>
                            </td>
                        </tr>
                        @endforeach
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
@endif
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        const subjectSelect = document.getElementById('subject_id');
        const teacherSelect = document.getElementById('teacher_id');

        function filterTeachers() {
            if (!subjectSelect || !teacherSelect) return;
            const selectedSubject = subjectSelect.value;
            Array.from(teacherSelect.options).forEach(opt => {
                if (!opt.value) return; // skip placeholder
                const subjects = (opt.dataset.subjects || '').split(',').filter(Boolean);
                const match = !selectedSubject || subjects.includes(selectedSubject);
                opt.style.display = match ? '' : 'none';
            });
            // If current selection is hidden, clear it
            const current = teacherSelect.selectedOptions[0];
            if (current && current.style.display === 'none') {
                teacherSelect.value = '';
            }
        }

        if (subjectSelect) {
            subjectSelect.addEventListener('change', filterTeachers);
            filterTeachers();
        }
    });
</script>
@endsection
